<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  User.staticemail
 *
 * @copyright   (C) 2020 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\User\AuthLog\Extension;

use Joomla\CMS\Form\Form;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Factory;
use Joomla\Utilities\ArrayHelper;
use Joomla\CMS\Environment\Browser;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\User\UserFactoryInterface;


// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * An example custom terms and conditions plugin.
 *
 * @since  3.9.0
 */
final class AuthLog extends CMSPlugin
{
    private $log = false;
    private $logged = false;
    protected $_user;
    /**
     * Load the language file on instantiation.
     *
     * @var    boolean
     * @since  3.1
     */
    protected $autoloadLanguage = true;    

    public function onContentPrepareData($context, $data)
    {
        if (!in_array($context, ['com_users.user','com_admin.profile','com_users.profile'])) {
            return true;
        }

        $app = Factory::getApplication();
        if(
            $app->isClient('site') && 
            (
                (int)$this->params->get('profile',0) === 0 || 
                $data->id !== $app->getIdentity()->id
            )
        ) {
            return true;
        }

        if(is_object($data)) {
            if(!isset($data->authlog) && $data->id??false > 0) {
                $data->authlog = array('authlog'=>$data->id);
                if(!HtmlHelper::isRegistered('users.authlog')) {
                    HtmlHelper::register('users.authlog',[__CLASS__,'authlog']);
                }
            }
        }

        return true;
    }

    public static function authlog($value) {
        return AuthLogHelper::display($value);
    }

    public function onContentPrepareForm($form, $data)
    {
        if(!($form instanceof Form)) {
            return true;
        }

        if (!in_array($form->getName(), ['com_users.user','com_users.profile','com_admin.profile'])) {
            return true;
        }
        $app = Factory::getApplication();
        $layout = $app->input->get('layout');
        if(
            $app->isClient('site') && 
            (
                $layout !== 'edit' || 
                (int)$this->params->get('profile',0) === 0 || 
                ArrayHelper::getValue((array)$data,'id') !== $app->getIdentity()->id
            )
        ) {
            return true;
        }

        Form::addFormPath(JPATH_PLUGINS.'/user/authlog/forms');
        $form->loadFile('authlog',false);

        return true;
    }

    public function onUserAfterDelete($user, $succes, $msg)
    {
        if(!$succes) {
            return true;
        }

        $userId = ArrayHelper::getValue($user, 'id', 0, 'int');
        if($userId === 0) {
            return true;
        }

        try
        {
            $db = Factory::getDbo();
            $query = $db->createQuery(true)
                ->delete('#__plg_user_authlog')
                ->where('user_id = ' . (int) $userId);
            $db->setQuery($query)->execute();
        }
        catch (\Exception $e)
        {
            Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
            return false;
        }

        return true;
    }

    public function onUserAfterLogin($options){
        $this->_log();
        return true;
    }

    public function onUserLoginFailure($credentials){
        $user = Factory::getUser($credentials['username']?:0);
        if(!$user->id) {
            return true;
        }
        $this->log = new \stdClass();
        $this->log->user_id = $user->id;
        $this->log->fail = 1; // bad password
        $this->_log();
        return true;
    }

    public function onUserAuthorisation($user, $options){
        $userfactory = Factory::getContainer()->get(UserFactoryInterface::class);
        $userobject = $userfactory->loadUserByUsername($user->username);
        $this->log = new \stdClass();
        $this->log->user_id = $userobject->id;
        $this->log->fail = 0;
        if($userobject->block){
            $this->log->fail = 2; // account disabled
            $this->_log();
            return;
        }
        if(!$userobject->authorise($options['action'])) {
            $this->log->fail = 3; // not authorized to log in
            $this->_log();
            return;
        }
    }

    private function _log(){
        if(!$this->log->user_id || $this->logged) {
            return true;
        }
        $db = Factory::getDbo();
        $browser = Browser::getInstance();
        $this->log->ip = $this->_getIP();
        $this->log->admin = Factory::getApplication()->isClient('administrator')?1:0;
        $this->log->mobile = $browser->isMobile()?1:0;
        $this->log->platform = $browser->getPlatform();
        $this->log->browser = $browser->getBrowser();
        $this->log->version = $browser->getVersion();
        $this->log->entry_url = getenv('REQUEST_URI');
        $db->insertObject('#__plg_user_authlog',$this->log);
        $this->logged = true;
        return true;
    }

    private function _getIP(){
        return getenv('HTTP_CLIENT_IP')?:
        getenv('HTTP_X_FORWARDED_FOR')?:
        getenv('HTTP_X_FORWARDED')?:
        getenv('HTTP_FORWARDED_FOR')?:
        getenv('HTTP_FORWARDED')?:
        getenv('REMOTE_ADDR');
    }
}