<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Task.inactiveusers
 *
 * @copyright   (C) 2023 Michael Richey. <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\Task\InactiveUsers\Extension;

use Joomla\CMS\Application\ApplicationHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Mail\Exception\MailDisabledException;
use Joomla\CMS\Mail\MailTemplate;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\User\UserFactoryAwareTrait;
use Joomla\CMS\User\UserHelper;
use Joomla\Component\Scheduler\Administrator\Event\ExecuteTaskEvent;
use Joomla\Component\Scheduler\Administrator\Task\Status;
use Joomla\Component\Scheduler\Administrator\Traits\TaskPluginTrait;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\ParameterType;
use Joomla\Event\SubscriberInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\Event\DispatcherAwareInterface;
use Joomla\Event\DispatcherAwareTrait;
use Joomla\CMS\Event\AbstractEvent;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * A task plugin. Offers 2 task routines Invalidate Expired Consents and Remind Expired Consents
 * {@see ExecuteTaskEvent}.
 *
 * @since 5.0.0
 */
final class InactiveUsers extends CMSPlugin implements SubscriberInterface
{
    use DatabaseAwareTrait;
    use TaskPluginTrait;
    use UserFactoryAwareTrait;

    protected $db;
    protected $dispatcher;

    /**
     * @var boolean
     *
     * @since 5.0.0
     */
    protected $autoloadLanguage = true;

    /**
     * @var string[]
     * @since 5.0.0
     */
    private const TASKS_MAP = [
        'check.deactivate' => [
            'langConstPrefix' => 'PLG_TASK_INACTIVEUSERS_DEACTIVATE',
            'method'          => 'deactivateUsers',
            'form'            => 'intervalForm',
        ],
        'check.warn' => [
            'langConstPrefix' => 'PLG_TASK_INACTIVEUSERS_WARN',
            'method'          => 'warnUsers',
            'form'            => 'intervalForm',
        ],
    ];

    /**
     * Constructor
     *
     * @param   DispatcherInterface              $dispatcher                 The object to observe
     * @param   array   $config    An array that holds the plugin configuration
     *
     * @since   1.5
     */
    public function __construct(
        DispatcherInterface $dispatcher,
        array $config
    )
    {
        parent::__construct($config);
    }

    /**
     * @inheritDoc
     *
     * @return string[]
     *
     * @since 5.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onTaskOptionsList'    => 'advertiseRoutines',
            'onExecuteTask'        => 'standardRoutineHandler',
            'onContentPrepareForm' => 'enhanceTaskItemForm',
        ];
    }

    /**
     * Method for the deactivateUsers task.
     *
     * @param   ExecuteTaskEvent  $event  The `onExecuteTask` event.
     *
     * @return integer  The routine exit code.
     *
     * @since  5.0.0
     * @throws \Exception
     */
    private function deactivateUsers(ExecuteTaskEvent $event): int
    {
        $ignoreGroups = $this->params->get('ignore_groups', []);
        $interval = $this->getIntervalFromEvent($event);
        $humanReadableInterval = $this->convertDurationToHumanReadable($event->getArgument('params')->interval ?? 'P1Y');
        $users = $this->getUsersByInterval($interval);

        if(!count($users)){
            return Status::OK;
        }
        $userFactory = $this->getUserFactory();
        $dispatcher = $this->getDispatcher();
        PluginHelper::importPlugin('inactiveusers', null, true, $dispatcher);
        $eventName = 'onInactiveUserDeactivate';

        foreach($users as $user){
            $userGroups = UserHelper::getUserGroups($user->id);

            // Check if user is in any ignored groups
            if (array_intersect($ignoreGroups, $userGroups)) {
                continue; // Skip this user
            }
            $jUser = $userFactory->loadUserById($user->id);
            if(!$jUser || $jUser->get('block') == 1){
                continue; // User already blocked or could not be loaded
            }

            // Deactivate the user
            $jUser->set('block', 1);
            $jUser->save();

            // Append note to user account
            $noteText = Text::sprintf('PLG_TASK_INACTIVEUSERS_DEACTIVATE_NOTE', (new \DateTime())->format('Y-m-d H:i:s'), $humanReadableInterval);
            $this->appendNoteToUser($user->id, 'PLG_TASK_INACTIVEUSERS_DEACTIVATE_NOTE_TITLE', $noteText);
            
            // dispatch event onInactiveUserDeactivate
            $data = ['userId' => $user->id, 'username' => $user->username, 'email' => $user->email];
            $dispatcher->dispatch( $eventName, AbstractEvent::create($eventName, ['subject'=>(object)['event'=>$eventName],'data'=>$data]));
        }

        return Status::OK;
    }

    /**
     * Method for the warnUsers task.
     *
     * @param   ExecuteTaskEvent  $event  The `onExecuteTask` event.
     *
     * @return integer  The routine exit code.
     *
     * @since  5.0.0
     * @throws \Exception
     */
    private function warnUsers(ExecuteTaskEvent $event): int
    {
        $ignoreGroups = $this->params->get('ignore_groups', []);
        $interval = $this->getIntervalFromEvent($event);
        $humanReadableInterval = $this->convertDurationToHumanReadable($event->getArgument('params')->interval ?? 'P1Y');
        $users = $this->getUsersByInterval($interval);

        if(!count($users)){
            return Status::OK;
        }
        $userFactory = $this->getUserFactory();
        $dispatcher = $this->getDispatcher();
        PluginHelper::importPlugin('inactiveusers', null, true, $dispatcher);
        $eventName = 'onInactiveUserWarn';

        foreach($users as $user){
            $userGroups = UserHelper::getUserGroups($user->id);

            // Check if user is in any ignored groups
            if (array_intersect($ignoreGroups, $userGroups)) {
                continue; // Skip this user
            }
            $jUser = $userFactory->loadUserById($user->id);
            if(!$jUser || $jUser->get('block') == 1){
                continue; // User already blocked or could not be loaded
            }
            
            // unlike deactivate, we do not block or message the user,
            // we just dispatch the event and let any listeners handle it

            // dispatch event onInactiveUserWarn
            $data = ['userId' => $user->id, 'username' => $user->username, 'email' => $user->email];
            $dispatcher->dispatch( $eventName, AbstractEvent::create($eventName, ['subject'=>(object)['event'=>$eventName],'data'=>$data]));
        }

        return Status::OK;
    }

    /**
     * convert ISO 8601 duration to Human Readable format
     * @param string $duration
     * @return string
     */
    private function convertDurationToHumanReadable(string $duration): string
    {
        $interval = new \DateInterval($duration);
        $parts = [];
        if ($interval->y > 0) {
            $parts[] = Text::plural('PLG_TASK_INACTIVEUSERS_YEAR', $interval->y);
        }
        if ($interval->m > 0) {
            $parts[] = Text::plural('PLG_TASK_INACTIVEUSERS_MONTH', $interval->m);
        }
        if ($interval->d > 0) {
            $parts[] = Text::plural('PLG_TASK_INACTIVEUSERS_DAY', $interval->d);
        }
        return implode(', ', $parts);
    }

    /**
     * Get the interval DateInterval object from the task event.
     *
     * @param   ExecuteTaskEvent  $event  The `onExecuteTask` event.
     *
     * @return \DateInterval
     *
     * @since 5.0.0
     * @throws \Exception
     */    
    private function getIntervalFromEvent(ExecuteTaskEvent $event): \DateInterval
    {
        $intervalSpec = $event->getArgument('params')->interval ?? 'P1Y';
        return new \DateInterval($intervalSpec);
    }

    /**
     * Retrieve the users using the provided interval
     * @param \DateInterval $interval
     * @return array<int, User>
     */
    private function getUsersByInterval(\DateInterval $interval): array
    {
        // special where clause: (lastvisitDate <= NOW() - $interval OR (registerdate <= NOW() - $interval AND lastvisitDate IS NULL))
        $specialWhere = '(' 
            . $this->db->quoteName('lastvisitDate') . ' <= ' . $this->db->quote((new \DateTime())->sub($interval)->format('Y-m-d H:i:s')) 
            . ' OR (' 
                . $this->db->quoteName('registerdate') . ' <= ' . $this->db->quote((new \DateTime())->sub($interval)->format('Y-m-d H:i:s')) 
                . ' AND ' 
                . $this->db->quoteName('lastvisitDate') . ' IS NULL'
            . ')'
        . ')';
        $query = $this->db->createQuery()
            ->select($this->db->quoteName(['id', 'lastvisitDate', 'email', 'name', 'username']))
            ->from($this->db->quoteName('#__users'))
            ->where($this->db->quoteName('block') . ' = 0')
            ->where($specialWhere);
            
        $this->db->setQuery($query);
        $users = $this->db->loadObjectList();
        if(count($users) === 0){
            return [];
        }
        return $users;
    }

    /**
     * Append note to user account
     * @param int $userId
     * @param string $noteText
     */
    private function appendNoteToUser(int $userId, string $subjectConstant, string $noteText): void
    {
        $note_catid = $this->params->get('note_catid', 0);
        // Implementation for appending note to user account
        $query = $this->db->createQuery()
            ->insert($this->db->quoteName('#__user_notes'))
            ->set($this->db->quoteName('user_id') . ' = ' . $this->db->quote($userId, ParameterType::INTEGER))
            ->set($this->db->quoteName('subject') . ' = ' . $this->db->quote(Text::_($subjectConstant), ParameterType::STRING))
            ->set($this->db->quoteName('body') . ' = ' . $this->db->quote($noteText))
            ->set($this->db->quoteName('state') . ' = ' . $this->db->quote(1, ParameterType::INTEGER))
            ->set($this->db->quoteName('created_time') . ' = NOW()')
            ->set($this->db->quoteName('modified_time') . ' = NOW()')
            ->set($this->db->quoteName('created_user_id') . ' = ' . $this->db->quote(0, ParameterType::INTEGER)); // System user
        if ($note_catid > 0) {
            $query->set($this->db->quoteName('catid') . ' = ' . $this->db->quote($note_catid, ParameterType::INTEGER));
        }
        $this->db->setQuery($query);
        try {
            $this->db->execute();
        } catch (\Exception $e) {
            error_log('Failed to append note to user ' . $userId . ': ' . $e->getMessage());
        }
    }
}