<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.voicesearch
 *
 * @copyright   (C) 2020 Michael Richey. <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\VoiceSearch\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Event\SubscriberInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Uri\Uri;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * System plugin to add additional security features to the administrator interface.
 *
 * @since  5.0.0
 */
final class VoiceSearch extends CMSPlugin
{
    protected $app;
    /**
     * Load the language file on instantiation.
     *
     * @var    boolean
     * @since  3.1
     */
    protected $autoloadLanguage = true;

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     *
     * @since   4.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onBeforeCompileHead' => 'onBeforeCompileHead'
        ];
    }

	function onBeforeCompileHead() {
        $this->app = Factory::getApplication();

        // normally I test the app client first, but since this only applies to html pages and admin
        // is optional, I'm testing the document type first
        $doc = $this->app->getDocument();
        if(($doc->getType()??false) !== 'html') {
            return;
        }

        if($this->app->isClient('administrator') && !$this->params->get('admin',0)) {
            return;
        }

        // prep for JS
        $debug = $this->app->getConfig()->get('debug',0);

        // loading language for javascript
        $this->loadLanguageStrings();

        // load the script
        $wa = $doc->getWebAssetManager();
        $wa->registerAndUseScript('plg_system_voicesearch', 'media/plg_system_voicesearch/js/voicesearch.'.($debug?'':'min.').'js',['version'=>'auto'], ['defer' => true, 'async' => true]);

		$options = $this->getScriptOptions();
        $options['debug'] = $debug;
        $doc->addScriptOptions('plg_system_voicesearch',$options);
    }

    private function getScriptOptions(){
        $targets = [];
        if($this->app->isClient('administrator')) {
            $targets[] = 'input[id=filter_search]';
        } else {
			$targets[] = 'input[id^=mod-finder-searchword]';
			$targets[] = 'input[id^=mod-search-searchword]';
			$targets[] = 'input[id=search-searchword]';
			$targets[] = '#finder-search input[id=q][name=q]';
        }
        foreach((array)$this->params->get('extrafields',array()) as $field) {
            $targets[] = $field->selector;
        }
		$options = array(
				'position'=>$this->params->get('position','before'),
				'lang'=>$this->app->getLanguage()->get('tag'),
				'buttonclass'=>$this->params->get('buttonclass','btn btn-success'),
				'buttonicon'=>$this->params->get('buttonicon',''),
                'buttontext'=>$this->params->get('textbuttons',1)?Text::_('PLG_SYSTEM_VOICESEARCH_LISTEN'):'',
				'buttonlisteningclass'=>$this->params->get('buttonlisteningclass','btn btn-danger'),
				'buttonlisteningicon'=>$this->params->get('buttonlisteningicon',''),
				'continuous'=>(bool)$this->params->get('continuous',0),
				'confidence'=>(float)$this->params->get('confidence',0.9),
				'interrimResults'=>(bool)$this->params->get('interrimResults',0),
				'autosubmit'=>(bool)$this->params->get('autosubmit',1),
                'targets'=>$targets
		);
        return $options;
    }

    private function loadLanguageStrings() {
        Text::script('PLG_SYSTEM_VOICESEARCH_LOWCONFIDENCE');
        Text::script('PLG_SYSTEM_VOICESEARCH_LISTEN');
        Text::script('PLG_SYSTEM_VOICESEARCH_LISTENSTOP');
        Text::script('PLG_SYSTEM_VOICESEARCH_BUTTON_TITLE');
    }
}
