<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.Markdown
 *
 * @copyright   (C) 2007 Michael Richey <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\System\Markdown\Extension;

use Joomla\CMS\Application\CMSWebApplicationInterface;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Cache\CacheController;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\Event\SubscriberInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Fig\Link\Link;

require __DIR__.'/vendor/autoload.php';

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * RicheyWeb Markdown Plugin.
 *
 * @since  1.5
 */
final class Markdown extends CMSPlugin
{
    protected $app;
    protected $doc;
    protected $tmpl;
    protected $current = false;

    /**
     * Cache instance.
     *
     * @var    CacheController
     * @since  1.5
     */
    private $_cache = null;

    /**
     * Cache controller factory interface
     *
     * @var    CacheControllerFactoryInterface
     * @since  4.2.0
     */
    private $cacheControllerFactory;

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     *
     * @since   4.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onAfterDispatch' => 'onAfterDispatch',
            // 'onBeforeCompileHead' => 'onBeforeCompileHead',
            'onAfterRender' => 'onAfterRender'
        ];
    }

    /**
     * Constructor
     *
     * @param   DispatcherInterface              $dispatcher                 The object to observe
     * @param   array   $config    An array that holds the plugin configuration
     *
     * @since   1.5
     */
    public function __construct(
        DispatcherInterface $dispatcher,
        array $config,
        CacheControllerFactoryInterface $cacheControllerFactory
    )
    {
        parent::__construct($dispatcher, $config);    
        if (!$this->app instanceof CMSWebApplicationInterface) {
            return false;
        }    
        $this->cacheControllerFactory = $cacheControllerFactory;
    }



    /**
     * This function is called after the application has been loaded
     * We use it to get the current URL and create the canonical URL
     */
    // public function onBeforeCompileHead(){
    public function onAfterDispatch(){
        // $this->app = Factory::getApplication();
        if (!$this->app->isClient('site')) {
            return true;
        }

        $this->doc = $this->app->getDocument();
        if($this->doc->getType() !== 'html') {
            return true;
        }

        $input = $this->app->input;
        $this->tmpl = $input->getCmd('tmpl', false);
        $this->current = Uri::getInstance();
        if(!$this->tmpl) {
            $this->doc->addHeadLink($this->cleanUrl('markdown'), 'alternate', 'rel', ['type' => 'text/markdown', 'data-asset-name' => 'System - Markdown']);
            $this->addHeaderLink('canonical');
            return true;
        }

        if($this->tmpl === 'markdown') {
            $this->addHeaderLink('markdown');
        }
        
        return true;
    }

    public function onAfterRender(){
        if (!$this->app->isClient('site')) {
            return true;
        }

        if($this->doc->getType() !== 'html') {
            return true;
        }

        if(!$this->tmpl){
            return true;
        }

        if($this->tmpl !== 'markdown'){
            return true;
        }
        
        // first order of business - get the current URL
        $cacheKey = md5($this->current);
        // initialize the cache
        if(!$this->_cache){
            $this->_cache = $this->cacheControllerFactory->createCacheController('output',['defaultgroup'=>'plg_system_markdown']);
            $this->_cache->setLifeTime((int)$this->app->get('cachetime', 60, 'INTEGER'));
            $this->_cache->setCaching(1);
        }
        if($this->_cache->contains($cacheKey)) {
            $markdown = $this->_cache->get($cacheKey);
            $this->deliver($markdown);
            return true;
        }
        $converter = new \League\HTMLToMarkdown\HtmlConverter();
        $converter->getEnvironment()->addConverter(new \League\HTMLToMarkdown\Converter\TableConverter());
        $converter->getConfig()->setOption('strip_tags', true);
        $converter->getConfig()->setOption('header_style', 'atx');

        $html = $this->app->getBody();
        $markdown = $converter->convert($html);
        $rootUrl = Uri::root(false);
        $markdown = preg_replace('/\]\(\//', ']('.$rootUrl, $markdown);
        $this->_cache->store($markdown, $cacheKey);
        $this->deliver($markdown);

    }
    
    // private function link($url, $rel) {
    //     $current .= (strstr($this->current,'?')?'&':'?') . 'tmpl=markdown';
    //     $this->doc->addHeadLink($current, 'alternate', 'rel', ['type' => 'text/markdown', 'data-asset-name' => 'System - Markdown']);
    // }

    private function cleanUrl($tmpl=false){
        $url = clone $this->current;
        $url->setVar('tmpl',$tmpl ?: null);
        return $url->toString();
    }

    /**
     * Add the canonical link to the header
     * The preload manager link method is private, so we need to use reflection to access it
     * but that's super easy, barely an inconvenience
     * you can't stop the signal 🖕
    */
    private function addHeaderLink($rel = 'canonical'){
        // $canonical = $this->cleanUrl($rel==='canonical'?false:$rel);
        $preloadManager = $this->doc->getPreloadManager();
        $linkProvider = $preloadManager->getLinkProvider();
        switch($rel){
            case 'canonical': // this is the canonical page
                $canonical = $this->cleanUrl(false);
                $url = $this->cleanUrl('markdown');
                $link = (new Link('alternate', $url))->withAttribute('type','text/markdown');
                $preloadManager->setLinkProvider($linkProvider->withLink($link));
                break;
            case 'markdown': // this is the markdown version of the page
                $links = $linkProvider->getLinksByRel('canonical');
                foreach($links as $link){
                    $linkProvider = $linkProvider->withoutLink($link);
                }

                $alternate = $linkProvider->getLinksByRel('alternate');
                foreach($alternate as $link){
                    $linkProvider = $linkProvider->withoutLink($link);
                }

                // $alternate = $this->cleanUrl('component');
                // $link = new Link('alternate', $alternate);
                // $linkProvider = $linkProvider->withLink($link);

                $canonical = $this->cleanUrl(false);
                $link = new Link('canonical',$canonical);
                $preloadManager->setLinkProvider($linkProvider->withLink($link));
                break;
        }
    }

    private function deliver($markdown)
    {
        $this->app->setHeader('Content-Type', 'text/markdown');
        $this->app->setBody($markdown);
    }
}
