<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.LinkCanonical
 *
 * @copyright   (C) 2007 Michael Richey <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\System\LinkCanonical\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Cache\CacheController;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Filesystem\File;
use ReflectionClass;
use ReflectionProperty;


// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * RicheyWeb Link Canonical Plugin.
 *
 * @since  1.5
 */
final class LinkCanonical extends CMSPlugin
{
    protected $app;
    protected $doc;
    private $current;
    private $canonical = false;
    private $prev = false;
    private $next = false;
    private $pagination = false;
    private $store = [];

    /**
     * Cache instance.
     *
     * @var    CacheController
     * @since  1.5
     */
    private $_cache = null;

    /**
     * Cache controller factory interface
     *
     * @var    CacheControllerFactoryInterface
     * @since  4.2.0
     */
    private $cacheControllerFactory;
    /**
     * Constructor
     *
     * @param   DispatcherInterface              $dispatcher                 The object to observe
     * @param   array   $config    An array that holds the plugin configuration
     *
     * @since   1.5
     */
    public function __construct(
        DispatcherInterface $dispatcher,
        array $config,
        CacheControllerFactoryInterface $cacheControllerFactory
    )
    {
        parent::__construct($dispatcher, $config);    
        if (!$this->app instanceof CMSWebApplicationInterface) {
            return false;
        }    
        $this->cacheControllerFactory = $cacheControllerFactory;
    }

    /**
     * This function is called after the application has been loaded
     * We use it to get the current URL and create the canonical URL
     */
    // public function onBeforeCompileHead(){
    public function onAfterDispatch(){
        $this->app = Factory::getApplication();
        if (!$this->app->isClient('site')) {
            return true;
        }
        $this->doc = $this->app->getDocument();
        switch($this->doc->getType()){
            case 'html':
                break;
            case 'feed':
                // get the current URL, strip format and type params from it, and set as canonical
                $this->current = Uri::getInstance();
                $query = $this->current->getQuery(true);
                foreach(['format', 'type','tmpl'] as $key) {
                    if(isset($query[$key])) {
                        unset($query[$key]);
                    }
                }
                if($query['lang'] === '*') {
                    unset($query['lang']);
                }
                $this->current->setQuery($query);
                $this->canonical = $this->current->toString();
                $this->addHeaderLink();
                return true;
            default:
                return true;
        }
        if($this->doc->getType() != 'html') {
            return true;
        }
        // check if we should ignore this page
        $task = $this->app->input->get->getCmd('task',false);
        if($task !== false) {
            return true;
        }
        // check if we should ignore this component
        $option = $this->app->input->getCmd('option',false);
        $options = $this->getOptionsArray();
        if (in_array($option, $options)) {
            return true;
        }
        
        // first order of business - get the current URL
        $this->current = Uri::getInstance();
        $cacheKey = md5($this->current->toString());

        // initialize the cache
        if(!$this->_cache){
            $this->_cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('output',['defaultgroup'=>'plg_system_linkcanonical']);
            $this->_cache->setCaching(1);
        }
        if($this->_cache->contains($cacheKey)) {
            $this->loadFromCache($cacheKey);
            return true;
        }

        // Obtaining the router allows us to get the current vars
        $router = $this->app->getRouter();
        $vars = array_filter(array_merge($router->getVars(), $this->current->getQuery(true)));
        
        // now we create the canonical URL using the current vars
        $scheme = $this->getScheme();
        $url = $scheme . '://' . $this->getHost();
        
        // conditionally including the port in the URL
        $port = $this->getPort($scheme);
        if ($port) {
            $url .= ':' . $port;
        }

        $url = $this->applyFix($url,$option,$vars);

        $urlParts = ['scheme', 'host', 'port', 'path', 'query'];
        $canonical = Uri::getInstance($url);
        // $query = $canonical->getQuery(true);
        $canonicalurl = $canonical->toString($urlParts);

        if($this->params->get('redirect',0,'BOOLEAN')) {
            $this->redirectCanonical($canonicalurl,$urlParts);
        }

        // trailing slash normalization - removing trailing slashes and trailing /index.php from $url
        $canonicalurl = $this->trailingSlash($canonicalurl);

        // store the canonical URL for later use
        $this->canonical = $canonicalurl;

        $this->addHeadLink();
        $this->addHeaderLink();
        $this->storeToCache($cacheKey);
    }

    /**
     * Applies a fix to the provided URL based on the given option and variables.
     *
     * @param string $url    The URL to be fixed or modified.
     * @param string $option The option or mode that determines how the fix is applied.
     * @param array  $vars   Additional variables or parameters used in the fix process.
     *
     * @return string The fixed or modified URL.
     */
    private function applyFix($url,$option,$vars){
        $fixespath = JPATH_PLUGINS.'/system/linkcanonical/src/Fixes/LinkCanonical_'.$option.'.php';
        if(file_exists($fixespath)) {
            // error_log(print_r($vars,true));
            $fixes = 'RicheyWeb\Plugin\System\LinkCanonical\Fixes\LinkCanonical_'.$option;
            $route = Route::_($fixes::fix($vars,$this->current,$this->pagination),false);
            if($route) {
                $url .= $route;
            } else {
                $url .= Route::_('index.php?'.http_build_query($vars),false);
            }
        } else {
            $url .= Route::_('index.php?'.http_build_query($vars),false);
        }

        return $url;
    }

    /**
     * Adds or removes a trailing slash from the given URL based on configuration or logic.
     *
     * @param string $url The URL to process.
     * @return string The processed URL with the appropriate trailing slash.
     */
    private function trailingSlash($url){
        $url = rtrim($url, '/');
        if (substr($url, -10) === '/index.php') {
            $url = substr($url, 0, -10);
        }
        return $url;
    }

    /**
     * Stores data to the cache using the provided cache key.
     *
     * @param string $cacheKey The key under which the data should be stored in the cache.
     *
     * @return void
     */
    private function storeToCache($cacheKey){
        $this->_cache->store(
            // [
            //     'canonical' => $this->canonical,
            //     'prev' => $this->prev,
            //     'next' => $this->next
            // ],
            $this->store,
            $cacheKey
        );
    }

    /**
     * Loads data from the cache using the specified cache key.
     *
     * @param string $cacheKey The key identifying the cached data to retrieve.
     * @return mixed The cached data if found, or false/null if not present.
     */
    private function loadFromCache($cacheKey){
        $canonicalSet = $this->_cache->get($cacheKey);
        // error_log(print_r($canonicalSet,true));
        $this->canonical = $canonicalSet['canonical'] ?? false;
        $this->prev = $canonicalSet['prev'] ?? false;
        $this->next = $canonicalSet['next'] ?? false;
        if(count($canonicalSet['unset']??[])){
            foreach($canonicalSet['unset'] as $canonicalKey) {
                unset($this->doc->_links[$canonicalKey]);
            }
        }
        if($this->canonical) {
            $this->link($this->canonical,'canonical');
            if($this->prev){
                if($this->prev == $this->canonical){
                    $this->customLink($this->prev,'prev');
                } else {
                    $this->link($this->prev,'prev');
                }
            }
            if($this->next){
                $this->link($this->next,'next');
            }
            $this->addHeaderLink();
        }
    }

    /**
     * Get the options array
     *
     * @return  array  The options array
     */
    private function getOptionsArray() {
        $ignore = (array)$this->params->get('ignore', [], 'ARRAY');
        $options = [];
        foreach($ignore as $item) {
            $options[] = trim($item->option);
        }
        return $options;
    }

    /**
     * Redirect the user to the canonical URL
     * after matching certain criteria
     *
     * @param   string  $url  The canonical URL
     */
    private function redirectCanonical($url,$urlParts) {
        // never redirect the homepage
        if($this->current->toString() == Uri::root()) {
            return;
        }

        // compare the current URL to the canonical URL, if no match - redirect
        $current = $this->current->toString($urlParts);
        // if the current url doesn't begin with the canonical url, redirect; this saves redirects on 
        // untracked query vars because the canonical will still be correct, but the current url will be different
        if(strpos($current,$url) !== 0) {
            $this->app->redirect($url);
        }
    }

    /**
     * Get the port for the URL
     *
     * @param   string  $scheme  The scheme of the URL
     *
     * @return  mixed  The port number or false
     */
    private function getPort($scheme) {
        // we're checking that the port matches the scheme.  If it doesn't, we need to include it in the URL
        $port = $this->current->getPort($scheme);
        switch($scheme) {
            case 'http':
                if((int)$port == 80) {
                    return false;
                }
                break;
            case 'https':
                if((int)$port == 443) {
                    return false;
                }
                break;
        }
        return $port;
    }

    /**
     * Get the host for the URL
     *
     * @return  string  The host
     */
    private function getHost() {
        $param = $this->params->get('host','');
        if(!strlen(trim($param))) {
            return $this->current->getHost();
        }
        return trim($param);
    }

    /**
     * Get the scheme for the URL
     *
     * @return  string  The scheme
     */
    private function getScheme() {
        $param = $this->params->get('scheme','0');
        if($param == '0') {
            return $this->current->getScheme();
        }
        return $param;
    }

    /**
     * Add the canonical link to the head
     */
    private function addHeadLink(){
        $headData = $this->doc->getHeadData();
        // error_log(print_r($headData,true));
        $canonicalKeys = $this->findCanonicalKeys($headData['links']);
        if($canonicalKeys !== false){
            $this->store['unset'] = $canonicalKeys;
            $override = (bool)$this->params->get('override',0,'INTEGER');
            if(!$override) {
                return;
            }
            foreach($canonicalKeys as $canonicalKey) {
                unset($this->doc->_links[$canonicalKey]);
            }
        }
        // $this->doc->addHeadLink($this->canonical, 'canonical','rel',['data-asset-name'=>'System - Link Canonical']);
        $this->link($this->canonical,'canonical');
        $this->store['canonical'] = $this->canonical;
        if(is_array($this->pagination) && count($this->pagination)){
            // error_log(print_r($this->pagination,true));
            if($this->pagination[0]??0 > 0) {
                $link = $this->appendPagination($this->canonical,$this->pagination[0]);
                $this->link($link, 'prev');
                $this->prev = $link;
                $this->store['prev'] = $link;
            }
            if($this->pagination[2] === 2) {
            //  this is so annoying - you can't have 2 different rels pointing to the same url...so I have to insert this one as a
            //  custom tag...FML
            //  $this->doc->addHeadLink($this->canonical, 'prev','rel',['data-asset-name'=>'System - Link Canonical']);
                $link = $this->appendPagination($this->canonical,false);
                $this->customLink($link, 'prev');
                $this->prev = $link;
                $this->store['prev'] = $link;
            }
            if(isset($this->pagination[1]) && $this->pagination[1]??false) {
                $link = $this->appendPagination($this->canonical,$this->pagination[1]);
                $this->link($link, 'next');
                $this->next = $link;
                $this->store['next'] = $link;
            }
        }
    }

    private function link($url, $rel) {
        $this->doc->addHeadLink($url, $rel, 'rel', ['data-asset-name' => 'System - Link Canonical']);
    }
    private function customLink($url, $rel) {
        $this->doc->addCustomTag('<link href="'.$url.'" rel="'.$rel.'" data-asset-name="System - Link Canonical" />');
    }

    private function appendPagination($url,$var){
        // die(print_r($this->pagination));
        $append = [];
        $input = $this->app->getInput();
        if(isset($this->pagination[3]) && is_array($this->pagination[3]) && count($this->pagination[3])){
            foreach($this->pagination[3] as $key=>$filter){
                $value = $input->get($key,null,$filter);
                if($value !== null){
                    $append[] = $key.'='.rawurlencode($value);
                }
            }
        }
        $glue = strpos($url,'?')===false?'?':'&';
        $o = parse_url($url.$glue.implode('&',$append));
        $user = isset($o['user']) ? $o['user'] : null;
        $pass = isset($o['pass']) ? $o['pass'] : null;
        $userpass = strlen($user.$pass)?$user.':'.$pass.'@':'';
        $port = isset($o['port']) ? ':'.$o['port'] : null;
        parse_str(($o['query']??''),$query);
        unset($query['start']);
        if(!count($query)) {
            return $o['scheme'].'://'.$userpass.$o['host'].$port.($o['path']??'').($var?'?start='.(int)$var:'');
        }
        if($var){
            $query['start'] = (int)$var;
        }
        $newquery = http_build_query($query);
        return $o['scheme'].'://'.$userpass.$o['host'].$port.$o['path'].'?'.$newquery;
    }

    private function findCanonicalKeys($links){
        $keys = [];
        foreach($links as $key => $link){
            if(!isset($link['relation'])) {
                continue;
            }
            if($link['relation'] === 'canonical'){
                $keys[] = $key;
            }
        }
        // error_log(print_r($keys,true));
        return count($keys)? $keys : false;
    }

    /**
     * Add the canonical link to the header
     * The preload manager link method is private, so we need to use reflection to access it
     * but that's super easy, barely an inconvenience
     * you can't stop the signal 🖕
    */
    private function addHeaderLink(){
        $preloadManager = $this->doc->getPreloadManager();
        $preloadReflection = new ReflectionClass($preloadManager);
        $linksMethod = $preloadReflection->getMethod('link');
        $linksMethod->setAccessible(true);
        $linksMethod->invokeArgs($preloadManager, [$this->canonical,'canonical']);
        if($this->prev) {
            $linksMethod->invokeArgs($preloadManager, [$this->prev,'prev']);
        }
        if($this->next) {
            $linksMethod->invokeArgs($preloadManager, [$this->next,'next']);
        }
    }
}
