<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.geoip
 *
 * @copyright   (C) 2020 Michael Richey. <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\GeoIP\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Event\SubscriberInterface;
use Joomla\CMS\Cache\CacheController;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Factory;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * System plugin to package multiple frontend scripts into single files to reduce requests.
 *
 * @since  5.0.0
 */
final class GeoIP extends CMSPlugin
{
    protected $app;
    protected $doc;
    protected $db;
    private $ip;
    private $country;

    /**
     * Cache instance.
     *
     * @var    CacheController
     * @since  1.5
     */
    private $_cache = null;

    /**
     * Cache controller factory interface
     *
     * @var    CacheControllerFactoryInterface
     * @since  4.2.0
     */
    private $cacheControllerFactory;
    /**
     * Constructor
     *
     * @param   DispatcherInterface              $dispatcher                 The object to observe
     * @param   array   $config    An array that holds the plugin configuration
     *
     * @since   1.5
     */
    public function __construct(
        DispatcherInterface $dispatcher,
        array $config,
        CacheControllerFactoryInterface $cacheControllerFactory
    )
    {
        parent::__construct($dispatcher, $config);
        // check if we are in the backend
        if ($this->app->isClient('administrator')) {
            return;
        }
        $this->doc = Factory::getDocument();
        if (!$this->doc instanceof \Joomla\CMS\Document\HtmlDocument) {
            return false;
        }
        $this->cacheControllerFactory = $cacheControllerFactory;
        $this->ip = $this->getIp();
        $this->country = null;
        if((bool)$this->params->get('always',1,'INTEGER')) {
            $this->getCountry();
        }
    }
     
    public function onBeforeCompileHead()
    {
        // check if we are in the backend
        if ($this->app->isClient('administrator')) {
            return;
        }

        // check if we are in the frontend
        if ($this->app->isClient('site') && $this->country){
            // check if we have a country code
            $this->doc->addScriptOptions('geoip.country', $this->country);
        }
    }

    /**
     * Method to get the IP address.
     *
     * @return  string|null  IP address.
     *
     * @since  5.0.0
     */
    public function getIp(): ?string
    {
        return getenv('HTTP_CLIENT_IP')?:
        getenv('HTTP_X_FORWARDED_FOR')?:
        getenv('HTTP_X_FORWARDED')?:
        getenv('HTTP_FORWARDED_FOR')?:
        getenv('HTTP_FORWARDED')?:
        getenv('REMOTE_ADDR');
    }

    /**
     * Method to get the country code.
     *
     * @return  string|null  Country code.
     *
     * @since  5.0.0
     */
    public function getCountry(): ?string
    {
        // have we already done this?
        if ($this->country !== null) {
            return $this->country;
        }

        // check user state
        $this->country = $this->app->getUserState('geoip.country', null);
        if ($this->country !== null) {
            return $this->country;
        }

        // check the cache
        if(!$this->_cache){
            $this->_cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('output',['defaultgroup'=>'plg_system_geoip']);
            $this->_cache->setCaching(1);
        }

        if($this->_cache->contains($this->ip)){
            $this->country = $this->_cache->get($this->ip);
        }

        // check the database
        if ($this->country === null) {
            $this->country = $this->getCountryFromIp($this->ip);
            $this->_cache->store($this->country, $this->ip);
        }

        $this->app->setUserState('geoip.country', $this->country);
        return $this->country;
    }

    /**
     * Method to get the country code from the IP address.
     *
     * @param   string  $ip  The IP address.
     *
     * @return  string|null  Country code.
     *
     * @since  5.0.0
     */
    private function getCountryFromIp(string $ip): ?string
    {
        $protocol = (strstr($ip, ':') !== false) ? 6 : 4;
        $query = $this->db->createQuery(true);
        $query->select('country')->from('#__geoip')->where('protocol = ' . (int)$protocol);
        switch($protocol) {
            case 4:
                $query->where('INET_ATON(' . $this->db->q($ip) .') BETWEEN start4 AND stop4');
                break;
            case 6:
                $query->where('INET6_ATON(' . $this->db->q($ip) .') BETWEEN start6 AND stop6');
                break;
        }
        $this->db->setQuery($query);
        $result = $this->db->loadResult();
        if ($result) {
            return $result;
        }
        return $this->params->get('default','US','STRING');
    }
}
