<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.expiresheaders
 *
 * @copyright   (C) 2007 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\ExpiresHeaders\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Date\Date;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Joomla! System Logging Plugin.
 *
 * @since  1.5
 */
final class ExpiresHeaders extends CMSPlugin
{
    protected $app;
    private $expire = false;
    private $itemid = null;
    private $activeitem = null;
    private $defaultitem = null;
    private $expireInfo = null;
    private $expired = false;
    private $itemrules = array();
    private $defaultrules = array();
    private $site = false;

    function __construct(DispatcherInterface $dispatcher, array $options = array()) {
        parent::__construct($dispatcher, $options);
        $this->app = Factory::getApplication();
        if($this->site = $this->app->isClient('site')) {
            $this->activeitem = $this->app->getMenu()->getActive();
            $this->defaultitem = $this->app->getMenu()->getDefault();
            $this->itemid = $this->activeitem?$this->activeitem->id:null;
            $this->getExpireItems();
            $this->getDefaults();
        }
    }

    function onAfterInitialise() {
        if ($this->site && $this->itemid) {
            $this->calcExpire();
            $this->expire($this->expireInfo);
        }
    }
    function onAfterRoute() {
        if ($this->site && !$this->expireInfo && !$this->expired) {
            $this->activeitem = $this->app->getMenu()->getActive();
            $this->itemid = $this->activeitem?$this->activeitem->id:null;
            $this->calcExpire();
            $this->expire($this->expireInfo);
        }
    }
    // function onAfterDispatch() {       
    //     if ($this->site && !$this->expireInfo && !$this->expired) {
    //         $this->activeitem = $this->app->getMenu()->getActive();
    //         $this->itemid = $this->activeitem?$this->activeitem->id:null;
    //         $this->calcExpire();
    //         error_log('expiring from onAfterDispatch: '.print_r($this->expireInfo,true));
    //         $this->expire($this->expireInfo);
    //     }
    // }

    function onAfterRender() {       
        if ($this->site && !$this->expired) {
            $this->expire($this->expireInfo);
        }
    }

    function calcExpire() {
        $uri = Uri::getInstance();
        // if the itemid is 0 and active menu is null, we're on the homepage - setting it to the default id from the menu
        if (
        // these first two indicate that we might be on the homepage where there is no itemid
                $this->itemid == 0 &&
                is_null($this->app->getMenu()->getActive()) &&
                // we passed the first two tests, now two more to confirm homepage
                (
                    $uri->current() == $uri->base() ||
                    $uri->current() == $this->defaultitem->link
                )
        ) {
            $this->itemid = $this->defaultitem->id;
            $this->app->getMenu()->setActive($this->itemid);
        }

        // does this page have a rule, and does that rule indicate modification
        if ($this->itemid && array_key_exists($this->itemid,$this->itemrules)) {
            $this->expireInfo = $this->itemrules[$this->itemid];
        } else {
            if($this->defaultrules['expires']) {
                $this->expireInfo = $this->defaultrules;
            }	
        }
    }

    function expire($vars) {
        if (!$vars) {
            return null;
        }
        if ($vars['expires']) {
            $date = new Date();
	        $date->setTimezone(new \DateTimeZone('GMT'));
            $expireheader = $date->setTimestamp($vars['time'])->format('D, d M Y H:i:s T');
            $this->app->allowCache(true);
            $this->app->setHeader('Expires', $expireheader,true);
        }
        if ($vars['cachecontrol']) {
	        $this->cacheControlHeader($vars);
        }
        if ($vars['pragma'] && $vars['publicprivate'] == 2) {
            $this->app->setHeader('Pragma', 'no-cache', true);
        }
        if ((int)$vars['caching']??0 === 1) {
            $this->app->set('caching', '0');
        }

        $this->expired = true;
        return true;
    }

    function cacheControlHeader($vars) {
        $valuearray = array();
        switch ($vars['publicprivate']) {
            case 0: $valuearray[] = 'public';
            break;
            case 1: $valuearray[] = 'private';
            break;
            case 2: $valuearray[] = 'no-cache';
            break;
        }
	
        $ppns = ($vars['publicprivate'] < 2 && $vars['nostore'] != 1)?$vars['seconds']:'0';
        $valuearray[] = 'max-age=' . $ppns;
        $valuearray[] = 'pre-check=' . $ppns;
        $valuearray[] = 'post-check=' . $ppns;
        
        if ($vars['nostore']) {
            $valuearray[] = 'no-store';
        }
        if ($vars['mustrevalidate']) {
            $valuearray[] = 'must-revalidate';
        }
    
        $this->app->setHeader('Cache-Control', implode(',', $valuearray), true);	
        }
    
        function getItemid() {
            return $this->activeitem?$this->activeitem->id:$this->defaultitem->id;
        }
        function getExpireItems(){
        $expireitems = (array)$this->params->get('expireitems',array());
        foreach($expireitems as $expireitem) {
            $data = array();
            foreach(get_object_vars($expireitem) as $property=>$value) {
            $name = str_replace('rule','',$property);
            switch($property) {
                case 'menulist':
                break;
                default:
                $data[$name]=$value?:0;
                break;
            }
            }
            $itemvars = $this->itemvars($data);
            foreach($expireitem->menulist as $itemid) {
            $this->itemrules[$itemid]=$itemvars;
            }
        }
    }

    function getDefaults(){
        $data = array();
        $fields = array(
            'defaultexpires'=>0,
            'defaultexpireslength'=>2,
            'defaultexpiresinterval'=>'hour',
            'defaultcachecontrol'=>1,
            'defaultpublicprivate'=>0,
            'defaultnostore'=>'',
            'defaultmustrevalidate'=>'',
            'defaultpragma'=>'',
            'defaultcaching'=>''
        );
        foreach($fields as $varname=>$default) {
	    $key = str_replace('default','',$varname);
            $data[$key] = $this->params->get($varname,$default);
        }
        $this->defaultrules=$this->itemvars($data);
    }
    
    function itemvars($data) {
        $ret = array();
        $ret['expires']=(bool)$data['expires'];
        $ret['time'] = strtotime('+' . $data['expireslength'] . ' ' . $data['expiresinterval']);
        $ret['seconds'] = $ret['time'] - time();
        $ret['expireslength'] = (int) $data['expireslength'];
        $ret['expiresinterval'] = $data['expiresinterval'];
        $ret['cachecontrol'] = (bool) $data['cachecontrol'];
        $ret['nostore'] = (array_key_exists('nostore',$data) && $data['nostore'] == '1') ? true : false;
        $ret['mustrevalidate'] = (array_key_exists('mustrevalidate',$data) && $data['mustrevalidate'] == '1') ? true : false;
        $ret['publicprivate'] = (int) $data['publicprivate'];
        $ret['pragma'] = (array_key_exists('pragma',$data) && $data['pragma'] == '1') ? true : false;
        $ret['caching'] = (array_key_exists('caching',$data) && $data['caching'] == '1') ? true : false;
        return $ret;
    }

}