<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.cleanfeed
 *
 * @copyright   (C) 2020 Michael Richey. <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\System\CleanFeed\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Event\SubscriberInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Factory;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * System plugin to add additional security features to the administrator interface.
 *
 * @since  5.0.0
 */
final class CleanFeed extends CMSPlugin
{
    protected $app;

    /**
     * Load the language file on instantiation.
     *
     * @var    boolean
     * @since  3.1
     */
    protected $autoloadLanguage = true;

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     *
     * @since   4.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onAfterRender' => 'onAfterRender'
        ];
    }

    public function onAfterRender()
    {
        $this->app = Factory::getApplication();
        if($this->app->isClient('administrator')) {
            return;
        }
        if($this->app->input->get('format',false) !== 'feed') {
            return;
        }
        $plugins = $this->params->get('plugins','*');
        if(strlen(trim($plugins)) === 0) {
            return;
        }
        $plugins = explode("\n",$plugins);
        $wildcard = array_search('*',$plugins);
        if($wildcard !== false) {
            $matches = [];
            $paired_tags_regex = '/\{(\w+)(?:.*?)\}(.*?)\{\/\1\}/s';
            $this->stripTags($paired_tags_regex);
            $single_tags_regex = '/\{(\w+)(?:.*?)\}/s';
            $this->stripTags($single_tags_regex);
        }
        foreach($plugins as $plugin) {
            if(trim($plugin) === '*') {
                continue;
            }
            $plugin = trim($plugin);
            if(strlen($plugin) === 0) {
                continue;
            }
            $regex = $this->buildRegex($plugin);
            $this->stripTags($regex);
        }
    }

    private function buildRegex($plugin){
        // first, detect if the plugin is a paired tag.  We do this by reversepairing the first and last characters,
        // and checking if the first character is contained within the string.
        $first = substr($plugin,0,1);
        $last = substr($plugin,-1);
        $paired = strpos($plugin,$last.$first) !== false;
        if($paired){
            // paired tags may contain content between the tags, so we need to account for that in the regex
            // we do this by splitting the tag into two parts, preg_quoting them, then putting them back together
            // with (.*?) in between.
            $parts = explode($last.$first,$plugin);
            $parts[0] .= $last;
            $parts[1] = $first.$parts[1];
            foreach($parts as $key=>$part){
                $parts[$key] = preg_quote($part,'/');
            }
            $parts[0] = str_replace(' args','\s(.*?)',$parts[0]);
            $regex = '/'.$parts[0].'(.*?)'.$parts[1].'/s';
            return $regex;
        }
        // if the tag is not paired, we can just preg_quote it and return the regex
        $regex = '/'.preg_quote($plugin,'/').'/s';
        $regex = str_replace(' args','\s(.*?)',$regex);
        return $regex;
    }

    private function stripTags($regex){
        $buffer = $this->app->getBody();
        $matches = [];
        preg_match_all($regex,$buffer,$matches);
        foreach($matches[0] as $match) {
            $buffer = str_replace($match,'',$buffer);
        }
        $this->app->setBody($buffer);
    }
}