<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  System.aimeta
 *
 * @copyright   (C) 2025 Michael Richey. <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\System\AIMeta\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Event\Event;
use Joomla\CMS\Event\Model\PrepareFormEvent;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Event\SubscriberInterface;
use Joomla\CMS\Event\Result\ResultAwareInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Factory;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\CMS\Language\Text;


// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Joomla! System Logging Plugin.
 *
 * @since  1.5
 */
final class AIMeta extends CMSPlugin
{
    protected $app;
    protected $db;
    protected $autoloadLanguage = true;
    protected $_cache = null;
    protected $caching = true;

    /**
     * The supported form contexts
     *
     * @var    array
     *
     * @since  3.9.0
     */
    protected $supportedContext = [
        'com_content.article',
        'com_contact.contact',
    ];

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return array
     *
     * @since   5.3.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onAjaxAimeta' => 'onAjaxAimeta',
            'onContentPrepareForm' => 'onContentPrepareForm'
        ];
    }

    public function onAjaxAimeta(Event $event)
    {
        $session = Factory::getSession();
        if($session->isNew()){
            throw new \RuntimeException('Session expired', 403);
        }
        $user = $session->get('user');
        if(empty($user) || !$user->id){
            throw new \RuntimeException('User not logged in', 403);
        }
        // this should be handled by the plugin ACL, so disabled for now unless someone convinces me otherwise
        // if(!$user->authorise('core.admin')){
        //     throw new \RuntimeException('User not authorized', 403);
        // }
        $data = [];
        $input = $this->app->input;
        $task = $input->getCmd('task', '');
        if(empty($task)){
            throw new \RuntimeException('No task specified', 400);
        }
        switch($task){
            case 'getModels':
                // setup cache for this task only
                if(!$this->_cache) {
                    $options = [
                        'defaultgroup' => 'plg_system_aimeta',
                        'caching' => true,
                        'lifetime' => $this->app->get('cachetime', 15),
                    ];
                    $this->caching = (bool)$this->params->get('caching', 1);
                    $this->_cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)
                        ->createCacheController('output',$options);
                }
                $data = $this->getModels($input);
                break;
            case 'getResponse':
                $data = $this->getResponse($input);
                $data = strip_tags($data);
                break;
            default:
                throw new \RuntimeException('Invalid task specified', 400);
        }
        $event->addResult($data);
    }

    public function onContentPrepareForm(PrepareFormEvent $event)
    {
        $form = $event->getForm();
        $name = $form->getName();

        if($name === 'com_plugins.plugin') {
            $input = $this->app->input;
            $extension_id = $this->getExtensionID();
            if($input->get('extension_id',0) == $this->getExtensionID()){
                $options = [
                    'metadesc_model' => $this->params->get('metadesc_model',''),
                    'metakey_model' => $this->params->get('metakeywords_model','')
                ];
                $this->app->getDocument()->addScriptOptions('plg_system_aimeta', $options);
            }
            return;
        }

        if (!in_array($name, $this->supportedContext, true)){
            return;
        }
        $ids = $this->getSourceElementIds();
        $options = [
            'context' => $name,
            'sourceElementIds' => $ids
        ];
        $doc = $this->app->getDocument();
        $doc->addScriptOptions('plg_system_aimeta', $options);
        $wa = $doc->getWebAssetManager();
        $wa->registerScript('plg_system_aimeta', 'media/plg_system_aimeta/js/aimeta.js', ['importmap' => true, 'version' => 'auto'], ['defer' => 'defer'], ['editors']);
        $wa->useScript('plg_system_aimeta');
        $initScript = '
            import plg_system_aimeta_class from "plg_system_aimeta";
            document.addEventListener("DOMContentLoaded", function() {
                if(typeof plg_system_aimeta_class !== "undefined") {
                    window.aimetaInstance = new plg_system_aimeta_class();
                }
            });';
        $doc->addScriptDeclaration($initScript,'module');
        Text::script('PLG_SYSTEM_AIMETA_REGEN_REASON_PROMPT');
        Text::script('PLG_SYSTEM_AIMETA_GENERATING');
        Text::script('PLG_SYSTEM_AIMETA_GENERATE');

        return true;
    }

    protected function getResponse($input){
        $field = $input->getCmd('field', '');
        if(empty($field)){
            throw new \RuntimeException('No field specified', 400);
        }
        $response = $this->generateResponse($input);
        if(empty($response)){
            throw new \RuntimeException('No response generated', 500);
        }
        return $response;
    }

    protected function generateResponse($input): string
    {
        $field = $input->getCmd('field','');
        $temperature = 0.7;
        switch($field){
            case 'jform_metadesc':
                $param = $this->params->get('metadesc',[]);
                $connection = $this->params->get('metadesc_connection','');
                $model = $this->params->get('metadesc_model','');
                $knowledge = $this->params->get('metadesc_knowledge',[]);
                $temperature = (float)$this->params->get('metadesc_temperature',0.7);
                break;
            case 'jform_metakey':
                $param = $this->params->get('metakeywords',[]);
                $connection = $this->params->get('metakeywords_connection','');
                $model = $this->params->get('metakeywords_model','');
                $knowledge = $this->params->get('metakeywords_knowledge',[]);
                $temperature = (float)$this->params->get('metakeywords_temperature',0.7);
                break;
            default:
                throw new \RuntimeException('Invalid field specified', 400);
        }

        $connections = (array)$this->params->get('connections', []);
        if(empty($connections)){
            throw new \RuntimeException('No connections configured', 500);
        }
        $connection = $this->getConnection($connection);
        if(!$connection){
            throw new \RuntimeException('Invalid connection specified', 500);
        }

        $replacements = [
            '%title%' => $input->json->getString('title','','RAW'),
            '%article%' => $input->json->getString('article','','RAW'),
            '%keywords%' => $input->json->getString('metakey','','RAW')
        ];
        $ids = $this->getSourceElementIds();
        if(!empty($ids)){
            foreach((array)$ids as $id){
                $replacements['%#'.$id.'%'] = $input->json->getString($id,'','RAW');
            }
        }

        $messages = [];

        foreach((array)$param->chat as $chat){
            $content = str_replace(array_keys($replacements), array_values($replacements), $chat->content);
            $messages[] = [
                'role' => $chat->role,
                'content' => $content
            ];
        }
        
        $current = $input->json->getString('current','','RAW');
        if(!empty($current)){
            $messages[] = [
                'role' => 'assistant',
                'content' => $current
            ];
            $reason = $input->json->getString('reason','','RAW');
            $messages[] = [
                'role' => 'user',
                'content' => $reason
            ];
        }
        $files = [];
        if(!empty($knowledge)){
            foreach((array)$knowledge as $know){
                $files[] = [
                    'type' => 'file',
                    'id' => $know->content
                ];
            }
        }
        
        switch($connection->type){
            case 'openai':
                // we're using curl because openai-php is fucking garbage
                $endpoint = $connection->url . '/chat/completions';
                $apiKey = $connection->apikey;
                if(empty($model)){
                    throw new \RuntimeException('No model specified', 400);
                }
                if(empty($apiKey)){
                    throw new \RuntimeException('No API key specified', 500);
                }
                $postData = [
                    'model' => $model,
                    'messages' => $messages,
                    'max_tokens' => 200,
                    'temperature' => (float)$temperature,
                ];
                if(!empty($files)){
                    $postData['files'] = $files;
                }
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $endpoint);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($ch, CURLOPT_POST, 1);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($postData));
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Authorization: Bearer ' . $apiKey,
                    'Content-Type: application/json'
                ]);
                $response = curl_exec($ch);
                if(curl_errno($ch)){
                    throw new \RuntimeException('Curl error: ' . curl_error($ch), 500);
                }
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                if($httpCode !== 200){
                    throw new \RuntimeException('OpenAI API error: ' . $response, $httpCode);
                }
                $responseData = json_decode($response, true);
                if(!isset($responseData['choices'][0]['message']['content'])){
                    throw new \RuntimeException('Invalid response from OpenAI API', 500);
                }
                return trim($responseData['choices'][0]['message']['content']);
            case 'ollama':
                include_once __DIR__.'/vendor/autoload.php';
                $client = \ArdaGnsrn\Ollama\Ollama::client($connection->url);
                $chatMessages = [];
                foreach((array)$messages as $msg){
                    $chatMessages[] = new \ArdaGnsrn\Ollama\Models\Message($msg['role'], $msg['content']);
                }
                $response = $client->chat()->create($param['model'], $chatMessages, [
                    'max_tokens' => 200,
                    'temperature' => $temperature,
                ]);
                if(empty($response->message)){
                    throw new \RuntimeException('No response from Ollama', 500);
                }
                return trim($response->message);
            default:
                throw new \RuntimeException('Unsupported connection type', 400);
        }

    }

    protected function getModels($input): array
    {
        $connectionCmd = $input->getCmd('connection', '');
        if(empty($connectionCmd)){
            throw new \RuntimeException('No connection specified', 400);
        }
        if($this->_cache->contains('models_'.$connectionCmd) && $this->caching){
            return $this->_cache->get('models_'.$connectionCmd);
        }
        if($input->get('url',false)) {
            // this is a plugin config request - so we get the connection info from input
            $connection = (object)[
                'type' => base64_decode($input->getCmd('type', '')),
                'apikey' => base64_decode($input->getString('apikey', '')),
                'url' => base64_decode($input->getString('url', '')),
                'connection' => $connectionCmd
            ];
        } else {
            $connection = $this->getConnection($connectionCmd);
        }
        if(!$connection){
            throw new \RuntimeException('Invalid connection specified', 400);
        }
        switch($connection->type){
            case 'openai':
                $data = $this->getOpenAIModels($connection);
                break;
            case 'ollama':
                $data = $this->getOllamaModels($connection);
                break;
            default:
                throw new \RuntimeException('Unsupported connection type', 400);
        }
        if(is_array($data) && empty($data)){
            throw new \RuntimeException('No models found', 500);
        }
        $this->cacheModels($connection->connection, $data);
        return $data;
    }

    protected function cacheModels($connectionName, $models){
        $cacheKey = 'models_'.$connectionName;
        $this->_cache->store($models, $cacheKey);
    }


    protected function getConnection($name){
        $connections = (array)$this->params->get('connections', []);
        foreach((array)$connections as $conn){
            if($conn->connection === $name){
                return $conn;
            }
        }
        return false;
    }

    protected function getOpenAIModels($connection): array
    {
        // we're using curl because openai-php is fucking garbage
        $endpoint = $connection->url . '/models';
        $apiKey = $connection->apikey;
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $endpoint);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $apiKey,
            'Content-Type: application/json'
        ]);
        $response = curl_exec($ch);
        if(curl_errno($ch)){
            throw new \RuntimeException('Curl error: ' . curl_error($ch), 500);
        }
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        if($httpCode !== 200){
            throw new \RuntimeException('OpenAI API error: ' . $response, $httpCode);
        }
        $responseData = json_decode($response, true);
        if(!isset($responseData['data']) || !is_array($responseData['data'])){
            throw new \RuntimeException('Invalid response from OpenAI API', 500);
        }
        $models = [];
        foreach((array)$responseData['data'] as $model){
            $models[] = $model['id'];
        }
        sort($models);
        return $models;
    }

    protected function getOllamaModels($connection): array
    {
        include_once __DIR__.'/vendor/autoload.php';
        $client = \ArdaGnsrn\Ollama\Ollama::client($connection->url);
        $response = $client->models()->list();
        if(!count($response->models)){
            throw new \RuntimeException('No models found', 500);
        }
        foreach((array)$response->models as $model){
            $models[] = $model->name;
        }
        sort($models);
        return $models;
    }

    protected function getSourceElementIds(){
        $ids = [];
        $metakey = (array)$this->params->get('metakeywords',[]);
        // error_log(print_r($metakey,true));
        if(!empty($metakey)){
            foreach((array)$metakey['chat'] as $key){
                preg_match_all('/%#(jform_.*?)%/',$key->content,$matches);
                if(isset($matches[1]) && !in_array($matches[1],$ids)){
                    $ids += $matches[1];
                }
            }
        }
        $metadesc = (array)$this->params->get('metadesc',[]);
        if(!empty($metadesc)){
            foreach((array)$metadesc['chat'] as $key){
                preg_match_all('/\%#(jform_.*?)\%/',$key->content,$matches);
                if(isset($matches[1]) && !in_array($matches[1],$ids)){
                    $ids += $matches[1];
                }
            }            
        }
        return array_unique($ids);
    }

    protected function getExtensionID(){
        $db = $this->db ?: Factory::getDbo();
        $query = $db->getQuery(true)
            ->select($db->quoteName('extension_id'))
            ->from($db->quoteName('#__extensions'))
            ->where($db->quoteName('type') . ' = ' . $db->quote('plugin'))
            ->where($db->quoteName('element') . ' = ' . $db->quote('aimeta'))
            ->where($db->quoteName('folder') . ' = ' . $db->quote('system'));
        $db->setQuery($query);
        return (int)$db->loadResult();
    }
}