/**
 * FilterTable v1.0.0
 *
 * A simple JavaScript library for filtering, sorting, and paginating HTML tables.
 * It provides a user-friendly interface for managing large datasets in a tabular format.
 * The library supports text, range, and enumeration filters, as well as sorting by column.
 * It also includes pagination controls to navigate through large datasets.
 *
 * Michael Richey
 * michael@richeyweb.com
 * https://www.richeyweb.com
 */

class FilterTable {
    constructor(tableId, config = {}) {
        this.table = document.getElementById(tableId);
        if (!this.table) throw new Error('Table not found');

        this.table.classList.add('table');

        const defaultLanguage = {
            searchLabel: "Search",
            rangeMinPlaceholder: "Min ({min})",
            rangeMaxPlaceholder: "Max ({max})",
            rangeSeparator: "to",
            enumAllOption: "All",
            rowsPerPageLabel: "Rows per page:",
            paginationPrevious: "Previous",
            paginationNext: "Next",
            paginationAriaLabel: "Table pagination",
            sortTitle: "Click to sort"
        };

        this.language = { ...defaultLanguage, ...(config.language || {}) };

        this.config = {
            sortHandleRow: config.sortHandleRow || 0,
            sortableColumns: config.sortableColumns || [],
            filterableColumns: config.filterableColumns || [],
            enumColumns: Number.isInteger(config.enumColumns) ? [config.enumColumns] : (Array.isArray(config.enumColumns) ? config.enumColumns : []),
            rangeColumns: config.rangeColumns || {},
            rowsPerPageOptions: config.rowsPerPageOptions || [5, 10, 25, 50],
            defaultSortColumn: config.defaultSortColumn !== undefined ? config.defaultSortColumn : null,
            defaultSortDirection: config.defaultSortDirection || 'asc'
        };

        this.headerRows = this.table.querySelectorAll('thead tr');
        if (this.config.sortHandleRow >= this.headerRows.length) {
            throw new Error('Specified sort handle row index exceeds available header rows');
        }

        this.sortHeaders = this.headerRows[this.config.sortHandleRow].querySelectorAll('th');
        this.allRows = Array.from(this.table.querySelectorAll('tbody tr'));
        this.currentPage = 1;
        this.rowsPerPage = this.config.rowsPerPageOptions[0];
        this.filteredRows = [...this.allRows];
        this.textFilterValue = '';
        this.currentSortColumn = this.config.defaultSortColumn;
        this.rangeFilters = {};
        this.enumValues = {};
        this.init();
    }

    init() {
        this.cleanup();
        this.calculateRangeExtremes();
        this.calculateEnumValues();
        this.addFilterForm();
        this.addFilterControls(); // Call filter controls first
        this.addSortControls(); // Then sort controls to layer on top
        this.addPaginationSelector();
        if (this.currentSortColumn !== null && this.config.sortableColumns.includes(this.currentSortColumn)) {
            this.sortTable(this.currentSortColumn, true);
        }
        this.updateDisplay();
    }

    cleanup() {
        const existingForm = this.table.previousElementSibling;
        if (existingForm && existingForm.tagName === 'FORM') {
            existingForm.remove();
        }
        const existingPagination = this.table.nextElementSibling;
        if (existingPagination && existingPagination.className.includes('d-flex')) {
            existingPagination.remove();
        }
    }

    calculateRangeExtremes() {
        Object.keys(this.config.rangeColumns).forEach(colIndex => {
            const values = this.allRows
                .map(row => parseFloat(row.cells[colIndex].textContent.trim()))
                .filter(val => !isNaN(val));
            const min = Math.min(...values);
            const max = Math.max(...values);
            const configuredMin = this.config.rangeColumns[colIndex].defaultMin;
            const configuredMax = this.config.rangeColumns[colIndex].defaultMax;
            const defaultMin = (configuredMin === undefined || configuredMin === -1) ? min : configuredMin;
            const defaultMax = (configuredMax === undefined || configuredMax === -1) ? max : configuredMax;
            this.rangeFilters[colIndex] = {
                min,
                max,
                currentMin: Math.max(min, Math.min(defaultMin, max)),
                currentMax: Math.min(max, Math.max(defaultMax, min))
            };
        });
    }

    calculateEnumValues() {
        this.config.enumColumns.forEach(colIndex => {
            const values = [...new Set(this.allRows
                .map(row => row.cells[colIndex].textContent.trim())
                .filter(val => val !== ''))];
            values.sort();
            this.enumValues[colIndex] = values;
        });
    }

    addFilterForm() {
        const form = document.createElement('form');
        form.className = 'mb-3';
        form.id = `${this.table.id}-filter-form`;
        form.addEventListener('submit', (e) => {
            e.preventDefault();
            e.stopPropagation();
        });

        const row = document.createElement('div');
        row.className = 'row g-3 align-items-center';

        if(this.config.filterableColumns.length) {
            const textCol = document.createElement('div');
            textCol.className = 'col-auto';
            const textGroup = document.createElement('div');
            textGroup.className = 'input-group input-group-sm';
            const textLabel = document.createElement('label');
            textLabel.className = 'input-group-text';
            textLabel.textContent = this.language.searchLabel;
            const textInput = document.createElement('input');
            textInput.type = 'text';
            textInput.className = 'form-control';
            textInput.placeholder = 'Filter table...';
            textInput.addEventListener('input', () => {
                this.textFilterValue = textInput.value.toLowerCase();
                this.applyFilters();
            });
            textGroup.appendChild(textLabel);
            textGroup.appendChild(textInput);
            textCol.appendChild(textGroup);
            row.appendChild(textCol);
        }

        this.config.enumColumns.forEach(colIndex => {
            const col = document.createElement('div');
            col.className = 'col-auto';
            const group = document.createElement('div');
            group.className = 'input-group input-group-sm';
            const label = document.createElement('label');
            label.className = 'input-group-text';
            label.textContent = this.sortHeaders[colIndex].textContent;
            group.appendChild(label);
            group.appendChild(this.createEnumFilter(colIndex));
            col.appendChild(group);
            row.appendChild(col);
        });

        Object.keys(this.config.rangeColumns).forEach(colIndex => {
            const col = document.createElement('div');
            col.className = 'col-auto';
            const group = document.createElement('div');
            group.className = 'input-group input-group-sm';

            const label = document.createElement('label');
            label.className = 'input-group-text';
            label.textContent = this.sortHeaders[colIndex].textContent;
            group.appendChild(label);

            const minInput = document.createElement('input');
            minInput.type = 'number';
            minInput.className = 'form-control';
            minInput.placeholder = this.language.rangeMinPlaceholder.replace('{min}', this.rangeFilters[colIndex].min);
            minInput.value = this.rangeFilters[colIndex].currentMin;
            minInput.addEventListener('input', () => {
                const value = minInput.value ? parseFloat(minInput.value) : this.rangeFilters[colIndex].min;
                this.rangeFilters[colIndex].currentMin = Math.max(this.rangeFilters[colIndex].min, Math.min(value, this.rangeFilters[colIndex].currentMax));
                minInput.value = this.rangeFilters[colIndex].currentMin;
                this.applyFilters();
            });
            group.appendChild(minInput);

            const toSeparator = document.createElement('span');
            toSeparator.className = 'input-group-text';
            toSeparator.textContent = this.language.rangeSeparator;
            group.appendChild(toSeparator);

            const maxInput = document.createElement('input');
            maxInput.type = 'number';
            maxInput.className = 'form-control';
            maxInput.placeholder = this.language.rangeMaxPlaceholder.replace('{max}', this.rangeFilters[colIndex].max);
            maxInput.value = this.rangeFilters[colIndex].currentMax;
            maxInput.addEventListener('input', () => {
                const value = maxInput.value ? parseFloat(maxInput.value) : this.rangeFilters[colIndex].max;
                this.rangeFilters[colIndex].currentMax = Math.min(this.rangeFilters[colIndex].max, Math.max(value, this.rangeFilters[colIndex].currentMin));
                maxInput.value = this.rangeFilters[colIndex].currentMax;
                this.applyFilters();
            });
            group.appendChild(maxInput);

            col.appendChild(group);
            row.appendChild(col);
        });

        form.appendChild(row);
        this.table.parentNode.insertBefore(form, this.table);
        this.filterForm = form;
    }

    createEnumFilter(colIndex) {
        const select = document.createElement('select');
        select.className = 'form-select form-select-sm';
        select.id = `${this.table.id}-filter-select-${colIndex}`;

        const allOption = document.createElement('option');
        allOption.value = '';
        allOption.textContent = this.language.enumAllOption;
        select.appendChild(allOption);

        this.enumValues[colIndex].forEach(value => {
            const option = document.createElement('option');
            option.value = value;
            option.textContent = value;
            select.appendChild(option);
        });

        select.addEventListener('change', () => this.applyFilters());
        return select;
    }

    addSortControls() {
        this.config.sortableColumns.forEach(colIndex => {
            const header = this.sortHeaders[colIndex];
            header.classList.add('text-center');

            // Preserve existing content (e.g., checkbox or original text)
            const existingContent = Array.from(header.childNodes);
            const originalText = header.textContent.trim();

            header.innerHTML = ''; // Clear to rebuild

            // Add text span with original text
            const textSpan = document.createElement('span');
            textSpan.textContent = originalText;
            textSpan.style.cursor = 'pointer';
            textSpan.title = this.language.sortTitle;
            textSpan.addEventListener('click', (e) => {
                e.stopPropagation();
                this.sortTable(colIndex);
            });
            header.appendChild(textSpan);

            // Add sort icon
            const sortIcon = document.createElement('i');
            if (colIndex === this.config.defaultSortColumn) {
                sortIcon.className = 'fas ' + 
                    (this.config.defaultSortDirection === 'asc' ? 'fa-sort-up' : 'fa-sort-down') + 
                    ' my-auto float-end';
            } else {
                sortIcon.className = 'fas fa-sort my-auto float-end';
            }
            sortIcon.style.cursor = 'pointer';
            sortIcon.title = this.language.sortTitle;
            sortIcon.addEventListener('click', (e) => {
                e.stopPropagation();
                this.sortTable(colIndex);
            });
            header.appendChild(sortIcon);

            // Reappend any existing checkbox (from addFilterControls)
            existingContent.forEach(node => {
                if (node.nodeName === 'INPUT' && node.type === 'checkbox') {
                    header.insertBefore(node, textSpan); // Checkbox before text
                }
            });
        });
    }

    addFilterControls() {
        this.config.filterableColumns.forEach(colIndex => {
            if (!this.config.enumColumns.includes(colIndex)) { // Skip enum columns
                const header = this.sortHeaders[colIndex];
                header.classList.add('text-center');

                // Preserve existing content (e.g., original text)
                const existingContent = Array.from(header.childNodes);
                const originalText = header.textContent.trim();

                // Only add checkbox if it doesn’t already exist
                if (!header.querySelector(`#${this.table.id}-filter-check-${colIndex}`)) {
                    header.innerHTML = ''; // Clear to rebuild

                    const checkbox = document.createElement('input');
                    checkbox.type = 'checkbox';
                    checkbox.className = 'form-check-input me-1 float-start';
                    checkbox.id = `${this.table.id}-filter-check-${colIndex}`;
                    checkbox.checked = true;
                    checkbox.addEventListener('change', (e) => {
                        e.stopPropagation();
                        this.applyFilters();
                    });
                    header.appendChild(checkbox);

                    // Add text span with original text
                    const textSpan = document.createElement('span');
                    textSpan.textContent = originalText;
                    header.appendChild(textSpan);

                    // Reappend any existing sort icon
                    existingContent.forEach(node => {
                        if (node.nodeName === 'I' && node.className.includes('fas')) {
                            header.appendChild(node);
                        }
                    });
                }
            }
        });
    }

    applyFilters() {
        const checkboxes = Array.from(document.querySelectorAll(`#${this.table.id}-filter-form ~ table#${this.table.id} thead input[type="checkbox"][id^="${this.table.id}-filter-check-"]`));
        const selects = Array.from(this.filterForm.querySelectorAll(`select[id^="${this.table.id}-filter-select-"]`));

        this.filteredRows = this.allRows.filter(row => {
            let show = true;

            if (this.textFilterValue) {
                let textMatch = false;
                let hasCheckedColumn = false;

                this.config.filterableColumns.forEach(colIndex => {
                    if (!this.config.enumColumns.includes(colIndex)) {
                        const checkbox = checkboxes.find(cb => cb.id === `${this.table.id}-filter-check-${colIndex}`);
                        if (checkbox && checkbox.checked) {
                            hasCheckedColumn = true;
                            const cellValue = row.cells[colIndex].textContent.toLowerCase();
                            if (cellValue.includes(this.textFilterValue)) {
                                textMatch = true;
                            }
                        }
                    }
                });

                show = show && (!hasCheckedColumn || textMatch);
            }

            this.config.enumColumns.forEach(colIndex => {
                const select = selects.find(sel => sel.id === `${this.table.id}-filter-select-${colIndex}`);
                if (select) {
                    const filterValue = select.value.toLowerCase();
                    if (filterValue) {
                        const cellValue = row.cells[colIndex].textContent.toLowerCase();
                        show = show && cellValue === filterValue;
                    }
                }
            });

            Object.keys(this.config.rangeColumns).forEach(colIndex => {
                const range = this.rangeFilters[colIndex];
                const cellValue = parseFloat(row.cells[colIndex].textContent.trim());
                if (!isNaN(cellValue)) {
                    if (cellValue < range.currentMin || cellValue > range.currentMax) {
                        show = false;
                    }
                }
            });

            return show;
        });

        this.currentPage = 1;
        this.updateDisplay();
    }

    sortTable(colIndex, isInitial = false) {
        const isNumeric = this.isColumnNumeric(colIndex);
        const header = this.sortHeaders[colIndex];
        let direction = isInitial 
            ? this.config.defaultSortDirection 
            : (header.dataset.sortDir === 'asc' ? 'desc' : 'asc');

        if (!isInitial) {
            this.config.sortableColumns.forEach(otherColIndex => {
                if (otherColIndex !== colIndex) {
                    const otherHeader = this.sortHeaders[otherColIndex];
                    const otherIcon = otherHeader.querySelector('i');
                    if (otherIcon) {
                        otherIcon.className = 'fas fa-sort my-auto float-end';
                        delete otherHeader.dataset.sortDir;
                    }
                }
            });
        }

        const sortIcon = header.querySelector('i');
        if (sortIcon) {
            sortIcon.className = 'fas ' + (direction === 'asc' ? 'fa-sort-up' : 'fa-sort-down') + ' my-auto float-end';
        }

        this.filteredRows.sort((a, b) => {
            const aValue = a.cells[colIndex].textContent.trim();
            const bValue = b.cells[colIndex].textContent.trim();
            if (isNumeric) {
                return direction === 'asc' ? Number(aValue) - Number(bValue) : Number(bValue) - Number(aValue);
            }
            return direction === 'asc' ? aValue.localeCompare(bValue) : bValue.localeCompare(aValue);
        });

        header.dataset.sortDir = direction;
        this.currentSortColumn = colIndex;
        this.currentPage = 1;
        this.updateDisplay();
    }

    isColumnNumeric(colIndex) {
        return this.allRows.every(row => {
            const value = row.cells[colIndex].textContent.trim();
            return !isNaN(value) && value !== '';
        });
    }

    addPaginationSelector() {
        const container = document.createElement('div');
        container.className = 'd-flex justify-content-between align-items-center mt-3';

        const label = document.createElement('label');
        label.className = 'me-2';
        label.textContent = this.language.rowsPerPageLabel;
        label.htmlFor = `${this.table.id}-rows-per-page`;

        const select = document.createElement('select');
        select.className = 'form-select form-select-sm w-auto';
        select.id = `${this.table.id}-rows-per-page`;

        this.config.rowsPerPageOptions.forEach(option => {
            const opt = document.createElement('option');
            opt.value = option;
            opt.textContent = option;
            if (option === this.rowsPerPage) opt.selected = true;
            select.appendChild(opt);
        });

        select.addEventListener('change', () => {
            this.rowsPerPage = parseInt(select.value);
            this.currentPage = 1;
            this.updateDisplay();
        });

        const paginationNav = document.createElement('nav');
        paginationNav.setAttribute('aria-label', this.language.paginationAriaLabel);
        const paginationUl = document.createElement('ul');
        paginationUl.className = 'pagination mb-0';
        paginationNav.appendChild(paginationUl);

        container.appendChild(label);
        container.appendChild(select);
        container.appendChild(paginationNav);

        this.table.parentNode.insertBefore(container, this.table.nextSibling);
        this.paginationContainer = container;
        this.paginationUl = paginationUl;
    }

    updatePagination() {
        if (!this.paginationUl) return;

        this.paginationUl.innerHTML = '';
        const pageCount = Math.ceil(this.filteredRows.length / this.rowsPerPage);

        if (this.filteredRows.length <= this.rowsPerPage) {
            this.paginationContainer.style.display = 'none';
            return;
        }

        this.paginationContainer.style.display = 'flex';

        const prevLi = document.createElement('li');
        prevLi.className = 'page-item' + (this.currentPage === 1 ? ' disabled' : '');
        const prevLink = document.createElement('a');
        prevLink.className = 'page-link';
        prevLink.href = '#';
        prevLink.textContent = this.language.paginationPrevious;
        prevLink.addEventListener('click', (e) => {
            e.preventDefault();
            if (this.currentPage > 1) {
                this.currentPage--;
                this.updateDisplay();
            }
        });
        prevLi.appendChild(prevLink);
        this.paginationUl.appendChild(prevLi);

        for (let i = 1; i <= pageCount; i++) {
            const li = document.createElement('li');
            li.className = 'page-item' + (i === this.currentPage ? ' active' : '');
            const link = document.createElement('a');
            link.className = 'page-link';
            link.href = '#';
            link.textContent = i;
            link.addEventListener('click', (e) => {
                e.preventDefault();
                this.currentPage = i;
                this.updateDisplay();
            });
            li.appendChild(link);
            this.paginationUl.appendChild(li);
        }

        const nextLi = document.createElement('li');
        nextLi.className = 'page-item' + (this.currentPage === pageCount ? ' disabled' : '');
        const nextLink = document.createElement('a');
        nextLink.className = 'page-link';
        nextLink.href = '#';
        nextLink.textContent = this.language.paginationNext;
        nextLink.addEventListener('click', (e) => {
            e.preventDefault();
            if (this.currentPage < pageCount) {
                this.currentPage++;
                this.updateDisplay();
            }
        });
        nextLi.appendChild(nextLink);
        this.paginationUl.appendChild(nextLi);
    }

    updateDisplay() {
        const tbody = this.table.querySelector('tbody');
        tbody.innerHTML = '';

        const start = (this.currentPage - 1) * this.rowsPerPage;
        const end = start + this.rowsPerPage;
        const pageRows = this.filteredRows.slice(start, end);

        pageRows.forEach((row, index) => {
            if (!row.dataset.originalIndex) {
                row.dataset.originalIndex = this.allRows.indexOf(row);
            }
            tbody.appendChild(row);
        });

        this.updatePagination();
    }
}

// Usage example
// const tableConfig = {
//     sortableColumns: [0], // Only ID sortable
//     filterableColumns: [1], // Only Name filterable
//     enumColumns: 2, // Gender as enum
//     rangeColumns: { 3: {} } // Age as range
// };

// new FilterTable('table1', tableConfig);