<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Content.ogre
 *
 * @copyright   (C) 2006 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\Content\MiniGlobe\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use RicheyWeb\Plugin\Content\MiniGlobe\Extension\TagParser;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Plugin to enable loading modules into content (e.g. articles)
 * This uses the {dialoglightbox} syntax
 *
 * @since  1.5
 */
final class MiniGlobe extends CMSPlugin
{
    protected $app;
    private $_tagparser = false;
    private $globes = 0;
    private $options = [
        'bumpmap' => false,
        'clouds' => false,
        'textureImage' => 'media/plg_content_miniglobe/images/default.webp',
        'bumpmapImage' => 'media/plg_content_miniglobe/images/bumpmap.webp',
        'cloudsImage' => 'media/plg_content_miniglobe/images/clouds.webp',
        'brightness' => 1,
        'speed' => 0.005,
        'hoverStop' => false,
        'markerColor' => [1.0, 0.0, 0.0],
        'markerSize' => 1,
        'markers' => '[]',
        'markersAJAX' => false,
        'markersKML' => false,
        'markersAJAXInterval' => 0,
        'oceanColor' => [0.05, 0.15, 0.35],
        'landColor' => [0.2, 0.45, 0.15],
        'fakeAtmosphere' => false,
        'graticule' => false,
        'gridColor' => [0.3, 0.3, 0.3],
        'gridThickness' => 0.0015,
        'lonInterval' => 30,
        'latInterval' => 15,
        'boldEquator' => true,
        'boldTropics' => false,
        'poleFade' => true,
        'startLongitude' => 0
    ];

    private $datasetMap = [
        'bumpmap' => false,
        'clouds' => false,
        'textureImage' => 'texture-image',
        'bumpmapImage' => false,
        'cloudsImage' => 'cloud-texture-image',
        'brightness' => false,
        'speed' => 'rotation-speed',
        'hoverStop' => 'pause-on-hover',
        'markerColor' => 'marker-color',
        'markerSize' => 'marker-size',
        'markerLength' => 'marker-length',
        'markers' => 'markers',
        'markersAJAX' => 'markers-ajax',
        'markersKML' => 'markers-kml',
        'markersAJAXInterval' => 'markers-ajax-interval',
        'oceanColor' => 'ocean-color',
        'landColor' => 'land-color',
        'fakeAtmosphere' => 'fake-atmosphere',
        'graticule' => 'graticule',
        'gridColor' => 'grid-color',
        'gridThickness' => 'grid-thickness',
        'lonInterval' => 'lon-interval',
        'latInterval' => 'lat-interval',
        'boldEquator' => 'bold-equator',
        'boldTropics' => 'bold-tropics',
        'poleFade' => 'pole-fade',
        'startLongitude' => 'start-longitude'
    ];

    private $conditionals = [
        'cloudsImage' => 'clouds',
    ];

    /**
     * Plugin that loads module positions within content
     *
     * @param   string   $context   The context of the content being passed to the plugin.
     * @param   object   &$article  The article object.  Note $article->text is also available
     * @param   mixed    &$params   The article params
     * @param   integer  $page      The 'page' number
     *
     * @return  void
     *
     * @since   1.6
     */
    public function onContentPrepare($context, &$article, &$params, $page = 0)
    {
        // Only execute in the site application
        if (!$this->app->isClient('site')) {
            return;
        }
        if(!strstr($article->text,'{miniglobe')) {
            return true;
        }
        if ($context == 'com_finder.indexer') {
            preg_replace('/{miniglobe(.*?)}/', '', $article->text);
            return true;
        }
        
        if(!$this->_tagparser){
            $this->_setup();
        }

        // strip out paragraph and div tags
        $regex = '/(<p.*?>)(\{miniglobe(.*?)\})(<\/p>)/';
        $article->text = preg_replace($regex, '$2', $article->text);

        // reset tags
        $this->_tagparser->tags = array();
        $this->_tagparser->parse($article->text);

        if (count($this->_tagparser->tags)) {
            foreach ($this->_tagparser->tags as $tag) {
                $replacement = $this->_setupGlobe($tag);
                $article->text = $tag->replace($article->text, $replacement);
            }
        }

    }

    private function _arrayToHtmlAttributes($array,$prefix=''){
        $attributes = array();
        foreach($array as $key=>$value){
            // escape quotes
            if(is_bool($value)){
                $value = $value ? 'true' : 'false';
            } elseif(is_array($value) || is_object($value)){
                $value = json_encode($value);
            } else {
                $value = (string)$value;
            }
            $value = htmlspecialchars($value, ENT_QUOTES);
            $attributes[] = $prefix.$key.'="'.$value.'"';
        }
        return implode(' ',$attributes);
    }

    private function _setupGlobe($tag){
        if($tag->getParam('loadonly',false)){
            return '';
        }
        $attributes = ['data-powered-by' => 'MiniGlobe by RicheyWeb (https://www.richeyweb.com)'];
        $html = '<div width="400" height="400" class="plg_content_miniglobe'.($tag->getParam('class',false)?' '.$tag->getParam('class',false):'').'">';
        $id = 'miniglobe_'.$this->globes;
        foreach($this->options as $option=>$default){
            // check conditionals
            $value = $tag->getParam($option,$default);
            if(!$this->_conditionalValue($option,$value,$tag)){
                continue;
            }
            $attribute = $this->datasetMap[$option];
            if(!$attribute) continue; // skip options where the $attribute value is false, these are no longer used
            $attributes[$attribute] = $this->_specialVars($option, $value, $tag);
        }
        $html .= '<canvas id="'.$id.'" width="100%" height="100%"class="miniglobe" '.$this->_arrayToHtmlAttributes($attributes, 'data-').'>';
        $this->globes++;
        $html .= '</canvas></div>';
        return $html;
    }

    private function _specialVars($option,$value,$tag){
        switch($option){
            case 'color':
            case 'markerColor':
            case 'oceanColor':
            case 'landColor':
            case 'gridColor':
                if(is_string($value) && str_starts_with($value,'[')){
                    $value = json_decode($value, true);
                }
                $rgb = $this->Hex2RGBDecimal($value);
                return json_encode($rgb);
            case 'image':
            case 'textureImage':
            case 'bumpmapImage':
            case 'cloudsImage':
                if($value === false || $value === 'false'){
                    return 'false';
                }
                return rtrim(Uri::root(true),'/').'/'.ltrim($value,'/');
            case 'markersAJAX':
            case 'markersKML':
                if($value === false || $value === 'false' || empty($value)){
                    return 'false';
                }
                return rtrim(Uri::root(true),'/').'/'.ltrim($value,'/');
            case 'fakeAtmosphere':
            case 'hoverStop':
            case 'graticule':
            case 'boldEquator':
            case 'boldTropics':
            case 'poleFade':
                if($value){
                    return 'true';
                } else {
                    return 'false';
                }
            default:
                return $value;
        }
    }

    private function _conditionalValue($option,$value,$tag) {
        if(isset($this->conditionals[$option])){
            $condition = $tag->getParam($this->conditionals[$option],$this->options[$this->conditionals[$option]]);
            return $condition;
        }
        return true;
    }

    private function _setup(){
        $this->_tagparser = new TagParser('miniglobe');
        $this->_tagparser->setOptions(array('selfclosing' => true));
        foreach($this->options as $option=>$default){
            $attribute = $this->datasetMap[$option];
            switch($option){
                case 'clouds':
                case 'hoverStop':
                    $this->options[$option] = $this->params->get($option,(bool)$default);
                    break;
                case 'markerSize':
                case 'brightness':
                case 'speed':
                    $this->options[$option] = $this->params->get($option,(float)$default);
                    break;
                case 'landColor':
                case 'oceanColor':
                case 'markerColor':
                    $color = $this->params->get($option,$default);
                    $rgb = $this->Hex2RGBDecimal($color);
                    $this->options[$option] = $rgb;
                    break;
                // case 'textureImage':
                // case 'bumpmapImage':
                // case 'cloudsImage':
                case 'markers':
                    break;
                default:
                    $this->options[$option] = Uri::root(true).$this->params->get($option,$default);
                    break;
            }
                    // error_log('Setting option '.$option.' to '.print_r($this->options[$option],true));
        }
        $doc = $this->app->getDocument();
        $mediaVersion = $doc->getMediaVersion();
        $wa = $doc->getWebAssetManager();
        $wa->registerAndUseStyle('plg_content_miniglobe', 'media/plg_content_miniglobe/css/miniglobe.css', [], ['version' => 'auto']);
        // $wa->registerAndUseScript('three.min', 'media/plg_content_miniglobe/js/three.min.js', [], ['version' => 'auto', 'defer' => true]);
        $wa->registerAndUseScript('plg_content_miniglobe', 'media/plg_content_miniglobe/js/miniglobe.js', [], ['version' => 'auto', 'defer' => true]);
    }

    private function Hex2RGBDecimal($hex) {
        if(is_array($hex)){
            return $hex;
        }
        if(strstr($hex,'rgb')){
            // already in rgb format
            $hex = str_replace(['rgb(',')',' '],'',$hex);
            $parts = explode(',',$hex);
            if(count($parts) == 3){
                $r = (int)$parts[0];
                $g = (int)$parts[1];
                $b = (int)$parts[2];
                $rv = [$r / 255, $g / 255, $b / 255]; 
                return $rv;
            } else {
                return [0, 0, 0]; // Invalid format
            }
        }   
        $hex = ltrim($hex, '#');
        switch(strlen($hex)){
            case 3:
                $r = hexdec(str_repeat(substr($hex, 0, 1), 2));
                $g = hexdec(str_repeat(substr($hex, 1, 1), 2));
                $b = hexdec(str_repeat(substr($hex, 2, 1), 2));
                break;
            case 6:
                $r = hexdec(substr($hex, 0, 2));
                $g = hexdec(substr($hex, 2, 2));
                $b = hexdec(substr($hex, 4, 2));
                break;
            default:
                return [0, 0, 0]; // Invalid format
        }
        $rv = [$r / 255, $g / 255, $b / 255]; 
        return $rv;
    }

}
