<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Contact.validemail
 *
 * @copyright   (C) 2007 Michael Richey. <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Joomla\Plugin\Contact\ValidEmail\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Event\Contact\SubmitContactEvent;
use Joomla\CMS\Cache\CacheController;
use Joomla\CMS\Cache\CacheControllerFactoryInterface;
use Joomla\Event\DispatcherInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\Utilities\ArrayHelper;
use Joomla\CMS\Application\SiteApplication;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Joomla! System Logging Plugin.
 *
 * @since  1.5
 */
final class ValidEmail extends CMSPlugin
{

    protected $app;

    /**
     * Cache instance.
     *
     * @var    CacheController
     * @since  1.5
     */
    private $_cache = null;

    /**
     * Load the language file on instantiation.
     *
     * @var    bool
     * @since  4.4.0
     */
    protected $autoloadLanguage = true;

    /**
     * Cache controller factory interface
     *
     * @var    CacheControllerFactoryInterface
     * @since  4.2.0
     */
    private $cacheControllerFactory;
    /**
     * Constructor
     *
     * @param   DispatcherInterface              $dispatcher                 The object to observe
     * @param   array   $config    An array that holds the plugin configuration
     *
     * @since   1.5
     */
    public function __construct(
        DispatcherInterface $dispatcher,
        array $config,
        CacheControllerFactoryInterface $cacheControllerFactory
    )
    {
        parent::__construct($dispatcher, $config);    
        // if (!$this->app instanceof CMSWebApplicationInterface) {
        if (!$this->app instanceof SiteApplication) {
            return false;
        }    
        $this->cacheControllerFactory = $cacheControllerFactory;
    }

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     *
     * @since   4.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return ['onSubmitContact' => 'onSubmitContact'];
    }

    function onSubmitContact(SubmitContactEvent $event){
        error_log('ValidEmail onSubmitContact called');
        $data        = $event->getData();
        $silent = (boolean)$this->params->get('silent', 0, 'INTEGER');
        
        // step 1: Check if the contact_email field is set
        if (!isset($data['contact_email']) || empty($data['contact_email'])) {
            if(!$silent) $this->app->enqueueMessage(Text::_('PLG_CONTACT_VALIDEMAIL_NO_EMAIL'), 'error');
            return;
        }
        
        // step 2: Validate the email address is syntactically correct
        if(!$this->filterValidateEmail($data['contact_email'])){
            if(!$silent) $this->app->enqueueMessage(Text::_('PLG_CONTACT_VALIDEMAIL_INVALID_ADDRESS'), 'error');
            $data['contact_email'] = '';
            $event->updateData($data);
            return;
        }
        
        // step 3: filter against black and white lists
        // first white email
        $we = $this->params->get('wemail', (new \stdClass));
        $wemail = $this->collectSubformValues($we,'email');
        if(count($wemail) && in_array(strtolower($data['contact_email']), $wemail)){
            return;
        }
        // now the white domains
        $wd = $this->params->get('wdomain', (new \stdClass));
        $wdomain = $this->collectSubformValues($wd,'domain');
        if(count($wdomain) && in_array(strtolower(substr(strrchr($data['contact_email'], "@"), 1)), $wdomain)){
            return;
        }
        // now the black email
        $be = $this->params->get('bemail', (new \stdClass));
        $bemail = $this->collectSubformValues($be,'email');
        if(count($bemail) && in_array(strtolower($data['contact_email']), $bemail)){
            // if the email is in the blacklist, we stop here
            if(!$silent) $this->app->enqueueMessage(Text::_('PLG_CONTACT_VALIDEMAIL_BLACKLISTED'), 'error');
            $data['contact_email'] = '';
            $event->updateData($data);
            return;
        }
        // now the black domains
        $bd = $this->params->get('bdomain', (new \stdClass));
        $bdomain = $this->collectSubformValues($bd,'domain');
        if(count($bdomain) && in_array(strtolower(substr(strrchr($data['contact_email'], "@"), 1)), $bdomain)){
            // if the email is in the blacklist, we stop here
            if(!$silent) $this->app->enqueueMessage(Text::_('PLG_CONTACT_VALIDEMAIL_BLACKLISTED'), 'error');
            $data['contact_email'] = '';
            $event->updateData($data);
            return;
        }
        
        // step 4: Validate the email address has an MX record
        $mx = true;
        if((int)$this->params->get('mx',1,'INTEGER') === 1) {
            $mx = $this->filterValidateMX($data['contact_email']);
        } else {
            // if we aren't checking MX, there isn't anything else to do
            return;
        }

        if (empty($mx)) {
            // if MX is false, there is no MX record for the domain
            if(!$silent) $this->app->enqueueMessage(Text::_('PLG_CONTACT_VALIDEMAIL_NO_MX_RECORD'), 'error');
            $data['contact_email'] = '';
            $event->updateData($data);
            return;
        }
        
        // step 5, verify that the MX isn't in the MX filter hosts
        $mxhosts = $this->params->get('mxhosts', (new \stdClass));
        $mxhosts = $this->collectSubformValues($mxhosts,'domain');
        if(!count($mxhosts)){
            // no MX hosts to filter against, so we are done
            return;
        }

        // $mxha = [];
        // foreach($mx as $mxhost){
        //     $mxhostarray = array_reverse(explode('.', strtolower($mxhost)));
        //     $mxha[] = $mxhostarray;
        // }
        // foreach($mxhosts as $mxhost){
        //     $hostarray = array_reverse(explode('.', strtolower($mxhost)));
        //     foreach($mxha as $mxh){
        //         $intersect = array_intersect($hostarray, $mxh);
        //         if($intersect === $hostarray){
        //             // if the MX host is in the MX filter hosts, we stop here
        //             if(!$silent) $this->app->enqueueMessage(Text::sprintf('PLG_CONTACT_VALIDEMAIL_MX_FILTERED', $mxhost), 'error');
        //             $data['contact_email'] = '';
        //             $event->updateData($data);
        //             return;
        //         }
        //     }
        // }
        if($this->MXBlacklisted(strtolower($mx[0]), $mxhosts)){
            // if the mx is in the blacklist, we stop here
            if(!$silent) $this->app->enqueueMessage(Text::sprintf('PLG_CONTACT_VALIDEMAIL_MX_FILTERED', $mx[0]), 'error');
            $data['contact_email'] = '';
            $event->updateData($data);
            return;
        }
        
        // If we get here, the email is allowed, syntactically valid, and has an MX record that is not blacklisted
    }

    private function MXBlacklisted($mx,$blacklist){
        // // Reverse MX hostname components
        // $mx_parts = array_reverse(explode('.', $mx));
        
        // // Check each blacklist entry
        // foreach ($blacklist as $entry) {
        //     $entry_parts = array_reverse(explode('.', $entry));
        //     if (array_intersect($entry_parts, $mx_parts) === $entry_parts) {
        //         return true;
        //     }
        // }

        // this performance rewrite eliminates looped array reversals and array_intersect
        // in favor of a single array_slice and comparison per loop iteration
        $mx_parts = explode('.',$mx);
        $mx_length = count($mx_parts);
        foreach($blacklist as $entry){
            $entry_parts = explode('.',$entry);
            $entry_length = count($entry_parts);
            if($mx_length < $entry_length){
                continue; // guard clause - if the mx is shorter than the entry, it can't match
            }
            $mx_slice = array_slice($mx_parts,-$entry_length);
            if($mx_slice === $entry_parts){
                return true;
            }
        }
        return false;
    }

    private function collectSubformValues($o,$col){
        $rv = [];
        foreach((array)$o as $item){
            $rv[] = strtolower($item->$col);
        }
        return $rv;
    }

    private function filterValidateEmail($email)
    {        
        if (filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return true;
        }
        return false;
    }

    private function filterValidateMX($email)
    {
        $domain = substr(strrchr($email, "@"), 1);
        // check the cache
        if(!$this->_cache){
            $this->_cache = Factory::getContainer()->get(CacheControllerFactoryInterface::class)->createCacheController('output',['defaultgroup'=>'plg_contact_validemail']);
            $this->_cache->setCaching(1);
        }

        if($this->_cache->contains($domain)){
            return json_decode($this->_cache->get($domain));
        }

        $mx = getmxrr($domain, $mxhosts);
        // clear empty results
        $mxhosts = array_filter($mxhosts??[], fn($v) => !empty($v));
        if(!count($mxhosts)){
            return false;
        }
        // if $mxhosts contains "0.0.0.0" we can assume the domain is invalid
        if(in_array('0.0.0.0',$mxhosts)){
            return false;
        }

        $this->_cache->store($domain, json_encode($mxhosts));

        return $mxhosts;
    }
}
