<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Console.indexingapi
 *
 * @copyright   (C) 2022 Michael Richey <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\Console\IndexingAPI\CliCommand;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\Console\Input\InputOption;
use Joomla\Console\Command\AbstractCommand;
use Joomla\CMS\Event\AbstractEvent;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\User\UserFactory;

class SubmitCliCommand extends AbstractCommand
{
	/**
	 * The default command name
	 *
	 * @var    string
	 * @since  4.0.0
	 */
	protected $app;
	protected $db;
	protected $dispatcher;
	protected static $defaultName = 'indexingapi:submit';
	private $symfonyStyle;

	/**
	 * Internal function to execute the command.
	 *
	 * @param   InputInterface   $input   The input to inject into the command.
	 * @param   OutputInterface  $output  The output to inject into the command.
	 *
	 * @return  integer  The command exit code
	 *
	 * @since   4.0.0
	 */
	protected function doExecute(InputInterface $input, OutputInterface $output): int
	{
		$this->symfonyStyle = new SymfonyStyle($input, $output);
		$this->symfonyStyle->title('Indexing API CLI');

		$url = $input->getOption('url') ?? null;
		
		if (empty($url))
		{
			$this->symfonyStyle->error('No URL provided. Use --url to provide a URL.');

			return 1;
		}

		$this->app = Factory::getApplication();
		$this->dispatcher = $this->app->getDispatcher();
		PluginHelper::importPlugin('content','indexingapi',true, $this->dispatcher);
		
		$this->symfonyStyle->info('Received item: ' . $url);

		$this->submitItem($url);

		$this->symfonyStyle->success('Indexing API completed');

		return 0;
	}

	/**
	 * Configure the command.
	 *
	 * @return  void
	 *
	 * @since   4.0.0
	 */
	protected function configure(): void
	{
		$this->setDescription('This command sends a request to the Indexing API for recently published content.');
		$this->setHelp(
			<<<EOF
The <info>%command.name%</info> command sends a request to the Indexing API for recently published content.
<info>php %command.full_name%</info>
EOF
		);

		$this->addOption('url', 'i', InputOption::VALUE_REQUIRED, 'The URL of the item to submit.');
	}

	/**
	 * Submit the items to the Indexing API.
	 *
	 * @param   string  $context  The context of the items.
	 * @param   array   $items    The items to submit.
	 *
	 * @return  void
	 *
	 * @since   4.0.0
	 */
	private function submitItem(string $url): void
	{
		$data = (object)[
			'link' => $url,
			'cli' => true,
			'state'=>1,
			'publish_up'=>'2024-01-01 00:00:00'
		];
		$contentEventArguments = [
			'context' => 'IndexingAPIExternalTrigger',
			'subject' => $data,
			'isNew' => false 
		];
		$event = AbstractEvent::create('onContentAfterSave', $contentEventArguments);
		$this->dispatcher->dispatch('onContentAfterSave', $event);
	}

	/**
	 * Configure the IO.
	 *
	 * @param   InputInterface   $input   The input to inject into the command.
	 * @param   OutputInterface  $output  The output to inject into the command.
	 *
	 * @return  void
	 */
	private function configureSymfonyIO(InputInterface $input, OutputInterface $output)
	{
	  $this->cliInput = $input;
	  $this->ioStyle  = new SymfonyStyle($input, $output);
	}
}