<?php

/**
 * @package     Joomla.Plugin
 * @subpackage  Console.indexingapi
 *
 * @copyright   (C) 2022 Michael Richey <https://www.richeyweb.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace RicheyWeb\Plugin\Console\IndexingAPI\CliCommand;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Joomla\Console\Command\AbstractCommand;
use Joomla\CMS\Event\AbstractEvent;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Registry\Registry;

class PublishedCliCommand extends AbstractCommand
{
	/**
	 * The default command name
	 *
	 * @var    string
	 * @since  4.0.0
	 */
	protected $app;
	protected $dispatcher;
	protected static $defaultName = 'indexingapi:published';
	private $symfonyStyle;
	
	/**
	 * Internal function to execute the command.
	 *
	 * @param   InputInterface   $input   The input to inject into the command.
	 * @param   OutputInterface  $output  The output to inject into the command.
	 *
	 * @return  integer  The command exit code
	 *
	 * @since   4.0.0
	 */
	protected function doExecute(InputInterface $input, OutputInterface $output): int
	{
		$this->symfonyStyle = new SymfonyStyle($input, $output);

		$this->symfonyStyle->title('Indexing API CLI');

		$contexts = [
			'com_content.article'=> 'getArticleRoute',
			'com_contact.contact'=> 'getContactRoute',
			'com_newsfeeds.newsfeed'=> 'getNewsfeedRoute',
			'com_tags.tag'=> 'getTagRoute',
			'com_weblinks.weblink'=> 'getWeblinkRoute'
		];

		$this->app = Factory::getApplication();
		$this->db = Factory::getDbo();
		$this->dispatcher = $this->app->getDispatcher();
		PluginHelper::importPlugin('content','indexingapi',true, $this->dispatcher);

		foreach(array_keys($contexts) as $context) {
			$this->symfonyStyle->section('Processing context: ' . $context);
			$items = $this->dbContextItems($context);
			$this->symfonyStyle->info('Found ' . count($items) . ' items');
			$this->submitItems($context,$items);
		}

		$this->symfonyStyle->success('Indexing API completed');

		return 0;
	}

	/**
	 * Configure the command.
	 *
	 * @return  void
	 *
	 * @since   4.0.0
	 */
	protected function configure(): void
	{
		$this->setDescription('This command sends a request to the Indexing API for recently published content.');
		$this->setHelp(
			<<<EOF
The <info>%command.name%</info> command sends a request to the Indexing API for recently published content.
<info>php %command.full_name%</info>
EOF
		);
	}

	/**
	 * Get the items for the given context.
	 *
	 * @param   string  $context  The context to get the items for.
	 *
	 * @return  array  The items for the given context.
	 *
	 * @since   4.0.0
	 */
	private function dbContextItems(string $context): array
	{
		$plugin = (array) PluginHelper::getPlugin('console', 'indexingapi');
		$params = new Registry($plugin['params']);
		$croninterval = $params->get('croninterval', 24, 'INTEGER');

		$items = [];
		$query = $this->db->createQuery(true);
		switch($context){
			case 'com_content.article':
				$query->select('a.id, a.catid, a.state, a.language, a.publish_up')
					->from('#__content as a')
					->where('a.state = 1')
					->where('a.access IN (5,1)') // guest and public
					->where('a.publish_up > a.created')
					->where('a.publish_up < NOW()')
					->where('a.publish_up > DATE_SUB(NOW(), INTERVAL '.$croninterval.' HOUR)')
					->order('a.publish_up DESC');
				$this->db->setQuery($query);
				$articles = $this->db->loadObjectList();
				foreach($articles as $article){
					$uri = \Joomla\Component\Content\Site\Helper\RouteHelper::getArticleRoute(
						$article->id,
						$article->catid,
						$article->language
					);
					$article->link = $uri;
					$article->cli = true;
					$items[] = $article;
				}
				break;
			case 'com_contact.contact':
				$query->select('a.id, a.catid, a.state, a.language, a.publish_up')
					->from('#__contact_details as a')
					->where('a.state = 1')
					->where('a.access IN (5,1)') // guest and public
					->where('a.publish_up > a.created')
					->where('a.publish_up < NOW()')
					->where('a.publish_up > DATE_SUB(NOW(), INTERVAL '.$croninterval.' HOUR)')
					->order('a.publish_up DESC');
				$this->db->setQuery($query);
				$contacts = $this->db->loadObjectList();
				foreach($contacts as $contact){
					$uri = \Joomla\Component\Contact\Site\Helper\RouteHelper::getContactRoute(
						$contact->id,
						$contact->catid,
						$contact->lang
					);
					$contact->link = $uri;
					$contact->cli = true;
					$items[] = $contact;
				}
                break;
			case 'com_newsfeeds.newsfeed':
				$query->select('a.id, a.catid, a.published, a.language, a.publish_up')
					->from('#__newsfeeds as a')
					->where('a.published = 1')
					->where('a.access IN (5,1)') // guest and public
					->where('a.publish_up > a.created')
					->where('a.publish_up < NOW()')
					->where('a.publish_up > DATE_SUB(NOW(), INTERVAL '.$croninterval.' HOUR)')
					->order('a.publish_up DESC');
				$this->db->setQuery($query);
				$newsfeeds = $this->db->loadObjectList();
				foreach($newsfeeds as $newsfeed){
					$uri = \Joomla\Component\Newsfeeds\Site\Helper\RouteHelper::getNewsfeedRoute(
						$newsfeed->id,
						$newsfeed->catid,
						$newsfeed->lang
					);
					$newsfeed->state = 1;
					$newsfeed->link = $uri;
					$newsfeed->cli = true;
					$items[] = $newsfeed;
				}
                break;
			case 'com_tags.tag':
				$query->select('a.id, a.alias, a.published, a.language, a.publish_up')
					->from('#__tags as a')
					->where('a.published = 1')
					->where('a.access IN (5,1)') // guest and public
					->where('a.publish_up > a.created_time')
					->where('a.publish_up < NOW()')
					->where('a.publish_up > DATE_SUB(NOW(), INTERVAL '.$croninterval.' HOUR)');
				$this->db->setQuery($query);
				$tags = $this->db->loadObjectList();
				foreach($tags as $tag){
					$uri = \Joomla\Component\Tags\Site\Helper\RouteHelper::getComponentTagRoute(
						$tag->id.':'.$tag->alias,
						$tag->language
					);
					$tag->state = 1;
					$tag->link = $uri;
					$tag->cli = true;
					$items[] = $tag;
				}
				break;
			case 'com_weblinks.weblink':
				// weblinks is a core extension, but not installed by default - verify that #__weblinks is present
				$this->db->setQuery('show tables');
				$weblinks = $this->db->loadColumn();
				$prefix = $this->db->getPrefix();
				if(!in_array($prefix.'weblinks', $weblinks)){
					$this->symfonyStyle->error('The weblinks table is not present. Skipping.');
					return [];
				}
				$query->select('a.id, a.catid, a.state, a.language, a.publish_up')
					->from('#__weblinks as a')
					->where('a.state = 1')
					->where('a.access IN (5,1)') // guest and public
					->where('a.publish_up > a.created')
					->where('a.publish_up < NOW()')
					->where('a.publish_up > DATE_SUB(NOW(), INTERVAL '.$croninterval.' HOUR)')
					->order('a.publish_up DESC');
				$this->db->setQuery($query);
				$weblinks = $this->db->loadObjectList();
				foreach($weblinks as $weblink){
					$uri = 'index.php?option=com_weblinks&view=weblink&id='.$weblink->id;
					$weblink->link = $uri;
					$weblink->cli = true;
					$items[] = $weblink;
				}
				break;
			default:
				$items = [];
		}
		return $items;
	}

	/**
	 * Submit the items to the Indexing API.
	 *
	 * @param   string  $context  The context of the items.
	 * @param   array   $items    The items to submit.
	 *
	 * @return  void
	 *
	 * @since   4.0.0
	 */
	private function submitItems(string $context, array $items): void
	{

		foreach($items as $key=>$item){
			$this->symfonyStyle->info('Submitting item: ' . $item->id);
			$contentEventArguments = [
				'context' => 'IndexingAPIExternalTrigger',
				'subject' => $item,
				'isNew' => false 
			];
			$event = AbstractEvent::create('onContentAfterSave', $contentEventArguments);
			$this->dispatcher->dispatch('onContentAfterSave', $event);
		}
	}
}