let HashCashClass = function(options) {
    const root = this;
    this.container = null;
    this.vars = {
        success: false,
        remote_addr: null,
        level: 3,
        request_time: null,
        iterations: null,
        delaystart: true,
        delaymonitor: false,
        listener: null,
        punish: false,
        cdp: false,
        trigger: false,
        hasUserInteraction: false,
        isCDPTrapped: false,
        nonce: false,
        nonceValue: null,
        noncefield: null  
    };    

    const construct = function(options) {
        if (!options || !options.container) {
            console.error('Invalid options provided to HashCashClass');
            return;
        }
        Object.assign(root.vars, options);
        root.container = document.getElementById(options.container);
        if (!root.container) {
            console.error('HashCash container not found:', options.container);
            return;
        }
        ['mousemove', 'keydown', 'click'].forEach(event => {
            document.addEventListener(event, (e) => { 
                if (e.isTrusted) {
                    root.vars.hasUserInteraction = true;
                }
            }, { 
                once: true
            });
        });
        if(root.vars.cdp){
            root.vars.isCDPTrapped = root.testWindowForCDPRuntime(window);
        }
        if (root.vars.nonce) { // Generate and add nonce if enabled
            root.vars.nonceValue = Math.random().toString(36).slice(2);
            root.vars.noncefield = document.createElement('input');
            root.vars.noncefield.type = 'hidden';
            root.vars.noncefield.name = 'hashcash_nonce';
            root.vars.noncefield.value = root.vars.nonceValue;
            root.container.form.appendChild(root.vars.noncefield);
        }
        if(!root.vars.delaystart){
            root.startCalculation();
        } else {
            root.vars.listener = root.container.form;
            root.vars.listener.addEventListener('change', root.handleChange);
        }
    }

    this.startCalculation = async function(event = null) {
        if (root.vars.success) return;
        window.dispatchEvent(new Event('plg_captcha_hashcash_started'));
        const checks = [
            /* event is NOT an instance of Event (therefor artificial) */
            (event) => Object.getPrototypeOf(event) !== Event.prototype,
            /* nonce check if enabled, otherwise normal event element value change test */
            (event) => root.vars.nonce ? (
                event.srcElement.value === event.srcElement.defaultValue && 
                event.srcElement.value !== root.vars.nonceValue
            /* event is not a real change event because nothing changed */
            ) : event.srcElement.value === event.srcElement.defaultValue,
            /* bot manipulated the nonce input */
            (event) => root.vars.nonce && event.srcElement === root.vars.noncefield,
            /* listener element is not the target (only the listener is allowed to send change) */
            (event) => event.currentTarget !== root.vars.listener,
            /* event is not bubbling (listener gets change through bubble) */
            (event) => event.bubbles === false,
            /* event is not trusted (is script initiated) */
            (event) => event.isTrusted === false,
            /* form has no interaction */
            () => root.vars.hasUserInteraction === false,
            /* test if the window object is trapped */
            () => root.vars.isCDPTrapped,
            /* test if the browser is a bot */
            () => ('webdriver' in navigator === true && navigator.webdriver !== false),
            /* playwright detection */
            () => (navigator.userAgent.includes("HeadlessChrome") || navigator.userAgent.includes("Playwright")),
            () => ('__playwright__binding__' in window),
            () => ('__pwInitScripts' in window)
        ];
        if (root.vars.delaystart && event && root.shuffle([...checks]).some(check => check(event))) {
            let eventType = Object.getPrototypeOf(event).constructor.name;
            if (root.vars.cdp && root.vars.isCDPTrapped) console.warn('CDP runtime detected for event:', eventType);
            if (root.vars.trigger) {
                // site operator has opted to trigger an event when a bot is detected
                let nonceresponse = root.vars.nonce && (event.srcElement.value === root.vars.nonceValue || event.srcElement === root.vars.noncefield);
                let detail = {
                    "hashcash_ip": root.vars.remote_addr,
                    "hashcash_time": root.vars.request_time,
                    "hashcash_ua": navigator.userAgent,
                    "hashcash_event_type": eventType, // attack type identifier
                    "hashcash_target": event.currentTarget.getAttribute('id') || 'unknown', // attack type identifier
                    "hashcash_bubbles":event.bubbles, // attack type identifier
                    "hashcash_trusted":event.isTrusted, // attack type identifier
                    "hashcash_url": window.location.href,
                    "hashcash_nonce": nonceresponse
                };
                window.dispatchEvent(new CustomEvent('plg_captcha_hashcash', {detail:detail}));
            }
            /* target element is not the listener and/or is not trusted (user initiated), so we do something else */
            if(root.vars.punish){
                // site operator has opted to inflict punishment on the bot attempting to trick the form
                root.vars.level = 32;
            } else {
                // site operator is a saint
                return;
            }
        } else if (root.vars.nonce && root.vars.noncefield) {
            // user passed bot check, remove nonce field
            root.container.form.removeChild(root.vars.noncefield);
            root.vars.noncefield = null; // Clear reference
        }
        root.vars.iterations = Math.min(Math.pow(100, root.vars.level), 1e9);
        try {
            let getCash = await root.getCash();
            const result = await getCash();
            root.container.disabled = false;
            root.container.value = result;
            root.vars.success = true;
            window.dispatchEvent(new Event('plg_captcha_hashcash_finished'));
        } catch (error) {
            console.error('Error generating hash:', error);
            window.dispatchEvent(new CustomEvent('plg_captcha_hashcash_error', {detail: error}));
        }
    }

    this.getCash = async function() {
        const text = root.vars.remote_addr + root.vars.request_time;
        let count = 0;

        return async function() {
            const pattern = new RegExp('^0{' + root.vars.level + '}');
            const MAX_ITERATIONS = 1e9; // Cap at 1 billion
            while (root.vars.iterations > 0 && count < MAX_ITERATIONS) {
                try {
                    // const hashBuffer = await root.sha256Subtle(text + count);
                    switch(root.vars.hash_algorithm) {
                        case 'PBKDF2':
                            var hashBuffer = await root.pbkdf2Subtle(text + count, 'JoomlaHashCashSalt', 1000, 'SHA-256', 32);
                            break;
                        case '256':
                        case '384':
                        case '512':
                            var hashBuffer = await root.shaSubtle(text + count);
                            break;
                        default:
                            throw new Error('Unsupported hash algorithm: ' + root.vars.hash_algorithm);
                    }
                    // const hashBuffer = await root.shaSubtle(text + count);
                    const hashString = Array.from(new Uint8Array(hashBuffer))
                        .map(b => b.toString(16).padStart(2, '0'))
                        .join('');
                    
                    if(pattern.test(hashString)){
                        return count;
                    }
                    count++;
                    root.vars.iterations--;
                } catch (error) {
                    console.error('Error generating hash:', error);
                    count++;
                    root.vars.iterations--;
                }
            }
            return false;
        };
    }

    this.shaSubtle = async function(text) {
        try {
            let algo = root.vars.hash_algorithm ?? 256;
            const encoder = new TextEncoder();
            const textBuffer = encoder.encode(text);
            const hashBuffer = await crypto.subtle.digest('SHA-' + algo, textBuffer);
            return hashBuffer;
        } catch (error) {
            throw new Error('Failed to generate SHA hash: ' + error.message);
        }
    }

    this.pbkdf2Subtle = async function(text, salt, iterations, hash = 'SHA-256', keyLength = 32) {
        try {
            const encoder = new TextEncoder();
            const textBuffer = encoder.encode(text);
            const saltBuffer = encoder.encode(salt);
            const key = await crypto.subtle.importKey(
                'raw',
                textBuffer,
                { name: 'PBKDF2' },
                false,
                ['deriveBits']
            );
            const params = {
                name: 'PBKDF2',
                hash: hash,
                salt: saltBuffer,
                iterations: iterations
            };
            const derivedBits = await crypto.subtle.deriveBits(
                params,
                key,
                keyLength * 8
            );
            return derivedBits;
        } catch (error) {
            throw new Error('Failed to generate PBKDF2 hash: ' + error.message);
        }
    }

    this.sha256Subtle = async function(text) {
        try {
            const encoder = new TextEncoder();
            const textBuffer = encoder.encode(text);
            const hashBuffer = await crypto.subtle.digest('SHA-256', textBuffer);
            return hashBuffer;
        } catch (error) {
            throw new Error('Failed to generate SHA256 hash: ' + error.message);
        }
    }

    this.testWindowForCDPRuntime = function(window) {
        if(!root.vars.cdp) return false;
        let trapped = false;
        const e = new Error();
        window.Object.defineProperty(e, 'stack', {
            configurable: false,
            enumerable: false,
            get: () => { trapped = true; return null; }
        });
        try {
            throw e;
        } catch (_) {
            // CDP might still access stack here
        }
        return trapped;
    }
    // Shuffle function (Fisher-Yates)
    this.shuffle = function(array) {
        for (let i = array.length - 1; i > 0; i--) {
            const j = Math.floor(Math.random() * (i + 1));
            [array[i], array[j]] = [array[j], array[i]];
        }
        return array;
    }
    this.handleChange = function(e) { root.startCalculation(e); };
    // Add cleanup method if needed
    this.destroy = function() {
        root.vars.listener.removeEventListener('change', root.handleChange);
    };
    construct(options);
};

// DOM load event listener
window.addEventListener('DOMContentLoaded', ()=>{
    new HashCashClass(Joomla.getOptions('plg_captcha_hashcash'));
});